<?php

namespace MathPHP\Probability\Distribution\Table;

use MathPHP\Exception;

/**
 * Table of χ² value vs P value
 *
 * Use degrees of freedom and desired P value to find χ² value.
 *
 * https://en.wikipedia.org/wiki/Chi-squared_distribution
 *
 * This table is provided only for completeness. It is common for statistics
 * textbooks to include this table, so this library does as well. It is better
 * to use the χ² distribution CDF function when a χ² value is required.
 *
 * Table values referenced from: https://www.medcalc.org/manual/chi-square-table.php
 */
class ChiSquared
{
    /**
     * χ² value vs P value table
     * Array with degrees of freedom as indexes.
     * Each degree of freedom index contains an array where the indexes are
     * P values, and the values are χ² values.
     *
     * [
     *     df => [p => χ², p => χ², ...],
     *     df => [p => χ², p => χ², ...],
     *     ...
     * ]
     * @var array
     */
    public const CHI_SQUARED_SCORES = [
        1    => ['0.995' => 0.0000393, '0.975' => 0.000982, '0.200' => 1.642,    '0.100' => 2.706,    '0.050' => 3.841,    '0.025' => 5.024,    '0.020' => 5.412,    '0.010' => 6.635,    '0.005' => 7.879,    '0.002' => 9.550,    '0.001' => 10.828],
        2    => ['0.995' => 0.0100,    '0.975' => 0.0506,   '0.200' => 3.219,    '0.100' => 4.605,    '0.050' => 5.991,    '0.025' => 7.378,    '0.020' => 7.824,    '0.010' => 9.210,    '0.005' => 10.597,   '0.002' => 12.429,   '0.001' => 13.816],
        3    => ['0.995' => 0.0717,    '0.975' => 0.216,    '0.200' => 4.642,    '0.100' => 6.251,    '0.050' => 7.815,    '0.025' => 9.348,    '0.020' => 9.837,    '0.010' => 11.345,   '0.005' => 12.838,   '0.002' => 14.796,   '0.001' => 16.266],
        4    => ['0.995' => 0.207,     '0.975' => 0.484,    '0.200' => 5.989,    '0.100' => 7.779,    '0.050' => 9.488,    '0.025' => 11.143,   '0.020' => 11.668,   '0.010' => 13.277,   '0.005' => 14.860,   '0.002' => 16.924,   '0.001' => 18.467],
        5    => ['0.995' => 0.412,     '0.975' => 0.831,    '0.200' => 7.289,    '0.100' => 9.236,    '0.050' => 11.070,   '0.025' => 12.833,   '0.020' => 13.388,   '0.010' => 15.086,   '0.005' => 16.750,   '0.002' => 18.907,   '0.001' => 20.515],
        6    => ['0.995' => 0.676,     '0.975' => 1.237,    '0.200' => 8.558,    '0.100' => 10.645,   '0.050' => 12.592,   '0.025' => 14.449,   '0.020' => 15.033,   '0.010' => 16.812,   '0.005' => 18.548,   '0.002' => 20.791,   '0.001' => 22.458],
        7    => ['0.995' => 0.989,     '0.975' => 1.690,    '0.200' => 9.803,    '0.100' => 12.017,   '0.050' => 14.067,   '0.025' => 16.013,   '0.020' => 16.622,   '0.010' => 18.475,   '0.005' => 20.278,   '0.002' => 22.601,   '0.001' => 24.322],
        8    => ['0.995' => 1.344,     '0.975' => 2.180,    '0.200' => 11.030,   '0.100' => 13.362,   '0.050' => 15.507,   '0.025' => 17.535,   '0.020' => 18.168,   '0.010' => 20.090,   '0.005' => 21.955,   '0.002' => 24.352,   '0.001' => 26.124],
        9    => ['0.995' => 1.735,     '0.975' => 2.700,    '0.200' => 12.242,   '0.100' => 14.684,   '0.050' => 16.919,   '0.025' => 19.023,   '0.020' => 19.679,   '0.010' => 21.666,   '0.005' => 23.589,   '0.002' => 26.056,   '0.001' => 27.877],
        10   => ['0.995' => 2.156,     '0.975' => 3.247,    '0.200' => 13.442,   '0.100' => 15.987,   '0.050' => 18.307,   '0.025' => 20.483,   '0.020' => 21.161,   '0.010' => 23.209,   '0.005' => 25.188,   '0.002' => 27.722,   '0.001' => 29.588],
        11   => ['0.995' => 2.603,     '0.975' => 3.816,    '0.200' => 14.631,   '0.100' => 17.275,   '0.050' => 19.675,   '0.025' => 21.920,   '0.020' => 22.618,   '0.010' => 24.725,   '0.005' => 26.757,   '0.002' => 29.354,   '0.001' => 31.264],
        12   => ['0.995' => 3.074,     '0.975' => 4.404,    '0.200' => 15.812,   '0.100' => 18.549,   '0.050' => 21.026,   '0.025' => 23.337,   '0.020' => 24.054,   '0.010' => 26.217,   '0.005' => 28.300,   '0.002' => 30.957,   '0.001' => 32.909],
        13   => ['0.995' => 3.565,     '0.975' => 5.009,    '0.200' => 16.985,   '0.100' => 19.812,   '0.050' => 22.362,   '0.025' => 24.736,   '0.020' => 25.472,   '0.010' => 27.688,   '0.005' => 29.819,   '0.002' => 32.535,   '0.001' => 34.528],
        14   => ['0.995' => 4.075,     '0.975' => 5.629,    '0.200' => 18.151,   '0.100' => 21.064,   '0.050' => 23.685,   '0.025' => 26.119,   '0.020' => 26.873,   '0.010' => 29.141,   '0.005' => 31.319,   '0.002' => 34.091,   '0.001' => 36.123],
        15   => ['0.995' => 4.601,     '0.975' => 6.262,    '0.200' => 19.311,   '0.100' => 22.307,   '0.050' => 24.996,   '0.025' => 27.488,   '0.020' => 28.259,   '0.010' => 30.578,   '0.005' => 32.801,   '0.002' => 35.628,   '0.001' => 37.697],
        16   => ['0.995' => 5.142,     '0.975' => 6.908,    '0.200' => 20.465,   '0.100' => 23.542,   '0.050' => 26.296,   '0.025' => 28.845,   '0.020' => 29.633,   '0.010' => 32.000,   '0.005' => 34.267,   '0.002' => 37.146,   '0.001' => 39.252],
        17   => ['0.995' => 5.697,     '0.975' => 7.564,    '0.200' => 21.615,   '0.100' => 24.769,   '0.050' => 27.587,   '0.025' => 30.191,   '0.020' => 30.995,   '0.010' => 33.409,   '0.005' => 35.718,   '0.002' => 38.648,   '0.001' => 40.790],
        18   => ['0.995' => 6.265,     '0.975' => 8.231,    '0.200' => 22.760,   '0.100' => 25.989,   '0.050' => 28.869,   '0.025' => 31.526,   '0.020' => 32.346,   '0.010' => 34.805,   '0.005' => 37.156,   '0.002' => 40.136,   '0.001' => 42.312],
        19   => ['0.995' => 6.844,     '0.975' => 8.907,    '0.200' => 23.900,   '0.100' => 27.204,   '0.050' => 30.144,   '0.025' => 32.852,   '0.020' => 33.687,   '0.010' => 36.191,   '0.005' => 38.582,   '0.002' => 41.610,   '0.001' => 43.820],
        20   => ['0.995' => 7.434,     '0.975' => 9.591,    '0.200' => 25.038,   '0.100' => 28.412,   '0.050' => 31.410,   '0.025' => 34.170,   '0.020' => 35.020,   '0.010' => 37.566,   '0.005' => 39.997,   '0.002' => 43.072,   '0.001' => 45.315],
        21   => ['0.995' => 8.034,     '0.975' => 10.283,   '0.200' => 26.171,   '0.100' => 29.615,   '0.050' => 32.671,   '0.025' => 35.479,   '0.020' => 36.343,   '0.010' => 38.932,   '0.005' => 41.401,   '0.002' => 44.522,   '0.001' => 46.797],
        22   => ['0.995' => 8.643,     '0.975' => 10.982,   '0.200' => 27.301,   '0.100' => 30.813,   '0.050' => 33.924,   '0.025' => 36.781,   '0.020' => 37.659,   '0.010' => 40.289,   '0.005' => 42.796,   '0.002' => 45.962,   '0.001' => 48.268],
        23   => ['0.995' => 9.260,     '0.975' => 11.689,   '0.200' => 28.429,   '0.100' => 32.007,   '0.050' => 35.172,   '0.025' => 38.076,   '0.020' => 38.968,   '0.010' => 41.638,   '0.005' => 44.181,   '0.002' => 47.391,   '0.001' => 49.728],
        24   => ['0.995' => 9.886,     '0.975' => 12.401,   '0.200' => 29.553,   '0.100' => 33.196,   '0.050' => 36.415,   '0.025' => 39.364,   '0.020' => 40.270,   '0.010' => 42.980,   '0.005' => 45.559,   '0.002' => 48.812,   '0.001' => 51.179],
        25   => ['0.995' => 10.520,    '0.975' => 13.120,   '0.200' => 30.675,   '0.100' => 34.382,   '0.050' => 37.652,   '0.025' => 40.646,   '0.020' => 41.566,   '0.010' => 44.314,   '0.005' => 46.928,   '0.002' => 50.223,   '0.001' => 52.620],
        26   => ['0.995' => 11.160,    '0.975' => 13.844,   '0.200' => 31.795,   '0.100' => 35.563,   '0.050' => 38.885,   '0.025' => 41.923,   '0.020' => 42.856,   '0.010' => 45.642,   '0.005' => 48.290,   '0.002' => 51.627,   '0.001' => 54.052],
        27   => ['0.995' => 11.808,    '0.975' => 14.573,   '0.200' => 32.912,   '0.100' => 36.741,   '0.050' => 40.113,   '0.025' => 43.195,   '0.020' => 44.140,   '0.010' => 46.963,   '0.005' => 49.645,   '0.002' => 53.023,   '0.001' => 55.476],
        28   => ['0.995' => 12.461,    '0.975' => 15.308,   '0.200' => 34.027,   '0.100' => 37.916,   '0.050' => 41.337,   '0.025' => 44.461,   '0.020' => 45.419,   '0.010' => 48.278,   '0.005' => 50.993,   '0.002' => 54.411,   '0.001' => 56.892],
        29   => ['0.995' => 13.121,    '0.975' => 16.047,   '0.200' => 35.139,   '0.100' => 39.087,   '0.050' => 42.557,   '0.025' => 45.722,   '0.020' => 46.693,   '0.010' => 49.588,   '0.005' => 52.336,   '0.002' => 55.792,   '0.001' => 58.301],
        30   => ['0.995' => 13.787,    '0.975' => 16.791,   '0.200' => 36.250,   '0.100' => 40.256,   '0.050' => 43.773,   '0.025' => 46.979,   '0.020' => 47.962,   '0.010' => 50.892,   '0.005' => 53.672,   '0.002' => 57.167,   '0.001' => 59.703],
        31   => ['0.995' => 14.458,    '0.975' => 17.539,   '0.200' => 37.359,   '0.100' => 41.422,   '0.050' => 44.985,   '0.025' => 48.232,   '0.020' => 49.226,   '0.010' => 52.191,   '0.005' => 55.003,   '0.002' => 58.536,   '0.001' => 61.098],
        32   => ['0.995' => 15.134,    '0.975' => 18.291,   '0.200' => 38.466,   '0.100' => 42.585,   '0.050' => 46.194,   '0.025' => 49.480,   '0.020' => 50.487,   '0.010' => 53.486,   '0.005' => 56.328,   '0.002' => 59.899,   '0.001' => 62.487],
        33   => ['0.995' => 15.815,    '0.975' => 19.047,   '0.200' => 39.572,   '0.100' => 43.745,   '0.050' => 47.400,   '0.025' => 50.725,   '0.020' => 51.743,   '0.010' => 54.776,   '0.005' => 57.648,   '0.002' => 61.256,   '0.001' => 63.870],
        34   => ['0.995' => 16.501,    '0.975' => 19.806,   '0.200' => 40.676,   '0.100' => 44.903,   '0.050' => 48.602,   '0.025' => 51.966,   '0.020' => 52.995,   '0.010' => 56.061,   '0.005' => 58.964,   '0.002' => 62.608,   '0.001' => 65.247],
        35   => ['0.995' => 17.192,    '0.975' => 20.569,   '0.200' => 41.778,   '0.100' => 46.059,   '0.050' => 49.802,   '0.025' => 53.203,   '0.020' => 54.244,   '0.010' => 57.342,   '0.005' => 60.275,   '0.002' => 63.955,   '0.001' => 66.619],
        36   => ['0.995' => 17.887,    '0.975' => 21.336,   '0.200' => 42.879,   '0.100' => 47.212,   '0.050' => 50.998,   '0.025' => 54.437,   '0.020' => 55.489,   '0.010' => 58.619,   '0.005' => 61.581,   '0.002' => 65.296,   '0.001' => 67.985],
        37   => ['0.995' => 18.586,    '0.975' => 22.106,   '0.200' => 43.978,   '0.100' => 48.363,   '0.050' => 52.192,   '0.025' => 55.668,   '0.020' => 56.730,   '0.010' => 59.893,   '0.005' => 62.883,   '0.002' => 66.633,   '0.001' => 69.346],
        38   => ['0.995' => 19.289,    '0.975' => 22.878,   '0.200' => 45.076,   '0.100' => 49.513,   '0.050' => 53.384,   '0.025' => 56.896,   '0.020' => 57.969,   '0.010' => 61.162,   '0.005' => 64.181,   '0.002' => 67.966,   '0.001' => 70.703],
        39   => ['0.995' => 19.996,    '0.975' => 23.654,   '0.200' => 46.173,   '0.100' => 50.660,   '0.050' => 54.572,   '0.025' => 58.120,   '0.020' => 59.204,   '0.010' => 62.428,   '0.005' => 65.476,   '0.002' => 69.294,   '0.001' => 72.055],
        40   => ['0.995' => 20.707,    '0.975' => 24.433,   '0.200' => 47.269,   '0.100' => 51.805,   '0.050' => 55.758,   '0.025' => 59.342,   '0.020' => 60.436,   '0.010' => 63.691,   '0.005' => 66.766,   '0.002' => 70.618,   '0.001' => 73.402],
        41   => ['0.995' => 21.421,    '0.975' => 25.215,   '0.200' => 48.363,   '0.100' => 52.949,   '0.050' => 56.942,   '0.025' => 60.561,   '0.020' => 61.665,   '0.010' => 64.950,   '0.005' => 68.053,   '0.002' => 71.938,   '0.001' => 74.745],
        42   => ['0.995' => 22.138,    '0.975' => 25.999,   '0.200' => 49.456,   '0.100' => 54.090,   '0.050' => 58.124,   '0.025' => 61.777,   '0.020' => 62.892,   '0.010' => 66.206,   '0.005' => 69.336,   '0.002' => 73.254,   '0.001' => 76.084],
        43   => ['0.995' => 22.859,    '0.975' => 26.785,   '0.200' => 50.548,   '0.100' => 55.230,   '0.050' => 59.304,   '0.025' => 62.990,   '0.020' => 64.116,   '0.010' => 67.459,   '0.005' => 70.616,   '0.002' => 74.566,   '0.001' => 77.419],
        44   => ['0.995' => 23.584,    '0.975' => 27.575,   '0.200' => 51.639,   '0.100' => 56.369,   '0.050' => 60.481,   '0.025' => 64.201,   '0.020' => 65.337,   '0.010' => 68.710,   '0.005' => 71.893,   '0.002' => 75.874,   '0.001' => 78.750],
        45   => ['0.995' => 24.311,    '0.975' => 28.366,   '0.200' => 52.729,   '0.100' => 57.505,   '0.050' => 61.656,   '0.025' => 65.410,   '0.020' => 66.555,   '0.010' => 69.957,   '0.005' => 73.166,   '0.002' => 77.179,   '0.001' => 80.077],
        46   => ['0.995' => 25.041,    '0.975' => 29.160,   '0.200' => 53.818,   '0.100' => 58.641,   '0.050' => 62.830,   '0.025' => 66.617,   '0.020' => 67.771,   '0.010' => 71.201,   '0.005' => 74.437,   '0.002' => 78.481,   '0.001' => 81.400],
        47   => ['0.995' => 25.775,    '0.975' => 29.956,   '0.200' => 54.906,   '0.100' => 59.774,   '0.050' => 64.001,   '0.025' => 67.821,   '0.020' => 68.985,   '0.010' => 72.443,   '0.005' => 75.704,   '0.002' => 79.780,   '0.001' => 82.720],
        48   => ['0.995' => 26.511,    '0.975' => 30.755,   '0.200' => 55.993,   '0.100' => 60.907,   '0.050' => 65.171,   '0.025' => 69.023,   '0.020' => 70.197,   '0.010' => 73.683,   '0.005' => 76.969,   '0.002' => 81.075,   '0.001' => 84.037],
        49   => ['0.995' => 27.249,    '0.975' => 31.555,   '0.200' => 57.079,   '0.100' => 62.038,   '0.050' => 66.339,   '0.025' => 70.222,   '0.020' => 71.406,   '0.010' => 74.919,   '0.005' => 78.231,   '0.002' => 82.367,   '0.001' => 85.351],
        50   => ['0.995' => 27.991,    '0.975' => 32.357,   '0.200' => 58.164,   '0.100' => 63.167,   '0.050' => 67.505,   '0.025' => 71.420,   '0.020' => 72.613,   '0.010' => 76.154,   '0.005' => 79.490,   '0.002' => 83.657,   '0.001' => 86.661],
        51   => ['0.995' => 28.735,    '0.975' => 33.162,   '0.200' => 59.248,   '0.100' => 64.295,   '0.050' => 68.669,   '0.025' => 72.616,   '0.020' => 73.818,   '0.010' => 77.386,   '0.005' => 80.747,   '0.002' => 84.943,   '0.001' => 87.968],
        52   => ['0.995' => 29.481,    '0.975' => 33.968,   '0.200' => 60.332,   '0.100' => 65.422,   '0.050' => 69.832,   '0.025' => 73.810,   '0.020' => 75.021,   '0.010' => 78.616,   '0.005' => 82.001,   '0.002' => 86.227,   '0.001' => 89.272],
        53   => ['0.995' => 30.230,    '0.975' => 34.776,   '0.200' => 61.414,   '0.100' => 66.548,   '0.050' => 70.993,   '0.025' => 75.002,   '0.020' => 76.223,   '0.010' => 79.843,   '0.005' => 83.253,   '0.002' => 87.507,   '0.001' => 90.573],
        54   => ['0.995' => 30.981,    '0.975' => 35.586,   '0.200' => 62.496,   '0.100' => 67.673,   '0.050' => 72.153,   '0.025' => 76.192,   '0.020' => 77.422,   '0.010' => 81.069,   '0.005' => 84.502,   '0.002' => 88.786,   '0.001' => 91.872],
        55   => ['0.995' => 31.735,    '0.975' => 36.398,   '0.200' => 63.577,   '0.100' => 68.796,   '0.050' => 73.311,   '0.025' => 77.380,   '0.020' => 78.619,   '0.010' => 82.292,   '0.005' => 85.749,   '0.002' => 90.061,   '0.001' => 93.168],
        56   => ['0.995' => 32.490,    '0.975' => 37.212,   '0.200' => 64.658,   '0.100' => 69.919,   '0.050' => 74.468,   '0.025' => 78.567,   '0.020' => 79.815,   '0.010' => 83.513,   '0.005' => 86.994,   '0.002' => 91.335,   '0.001' => 94.461],
        57   => ['0.995' => 33.248,    '0.975' => 38.027,   '0.200' => 65.737,   '0.100' => 71.040,   '0.050' => 75.624,   '0.025' => 79.752,   '0.020' => 81.009,   '0.010' => 84.733,   '0.005' => 88.236,   '0.002' => 92.605,   '0.001' => 95.751],
        58   => ['0.995' => 34.008,    '0.975' => 38.844,   '0.200' => 66.816,   '0.100' => 72.160,   '0.050' => 76.778,   '0.025' => 80.936,   '0.020' => 82.201,   '0.010' => 85.950,   '0.005' => 89.477,   '0.002' => 93.874,   '0.001' => 97.039],
        59   => ['0.995' => 34.770,    '0.975' => 39.662,   '0.200' => 67.894,   '0.100' => 73.279,   '0.050' => 77.931,   '0.025' => 82.117,   '0.020' => 83.391,   '0.010' => 87.166,   '0.005' => 90.715,   '0.002' => 95.140,   '0.001' => 98.324],
        60   => ['0.995' => 35.534,    '0.975' => 40.482,   '0.200' => 68.972,   '0.100' => 74.397,   '0.050' => 79.082,   '0.025' => 83.298,   '0.020' => 84.580,   '0.010' => 88.379,   '0.005' => 91.952,   '0.002' => 96.404,   '0.001' => 99.607],
        61   => ['0.995' => 36.301,    '0.975' => 41.303,   '0.200' => 70.049,   '0.100' => 75.514,   '0.050' => 80.232,   '0.025' => 84.476,   '0.020' => 85.767,   '0.010' => 89.591,   '0.005' => 93.186,   '0.002' => 97.665,   '0.001' => 100.888],
        62   => ['0.995' => 37.068,    '0.975' => 42.126,   '0.200' => 71.125,   '0.100' => 76.630,   '0.050' => 81.381,   '0.025' => 85.654,   '0.020' => 86.953,   '0.010' => 90.802,   '0.005' => 94.419,   '0.002' => 98.925,   '0.001' => 102.166],
        63   => ['0.995' => 37.838,    '0.975' => 42.950,   '0.200' => 72.201,   '0.100' => 77.745,   '0.050' => 82.529,   '0.025' => 86.830,   '0.020' => 88.137,   '0.010' => 92.010,   '0.005' => 95.649,   '0.002' => 100.182,  '0.001' => 103.442],
        64   => ['0.995' => 38.610,    '0.975' => 43.776,   '0.200' => 73.276,   '0.100' => 78.860,   '0.050' => 83.675,   '0.025' => 88.004,   '0.020' => 89.320,   '0.010' => 93.217,   '0.005' => 96.878,   '0.002' => 101.437,  '0.001' => 104.716],
        65   => ['0.995' => 39.383,    '0.975' => 44.603,   '0.200' => 74.351,   '0.100' => 79.973,   '0.050' => 84.821,   '0.025' => 89.177,   '0.020' => 90.501,   '0.010' => 94.422,   '0.005' => 98.105,   '0.002' => 102.691,  '0.001' => 105.988],
        66   => ['0.995' => 40.158,    '0.975' => 45.431,   '0.200' => 75.424,   '0.100' => 81.085,   '0.050' => 85.965,   '0.025' => 90.349,   '0.020' => 91.681,   '0.010' => 95.626,   '0.005' => 99.330,   '0.002' => 103.942,  '0.001' => 107.258],
        67   => ['0.995' => 40.935,    '0.975' => 46.261,   '0.200' => 76.498,   '0.100' => 82.197,   '0.050' => 87.108,   '0.025' => 91.519,   '0.020' => 92.860,   '0.010' => 96.828,   '0.005' => 100.554,  '0.002' => 105.192,  '0.001' => 108.526],
        68   => ['0.995' => 41.713,    '0.975' => 47.092,   '0.200' => 77.571,   '0.100' => 83.308,   '0.050' => 88.250,   '0.025' => 92.689,   '0.020' => 94.037,   '0.010' => 98.028,   '0.005' => 101.776,  '0.002' => 106.440,  '0.001' => 109.791],
        69   => ['0.995' => 42.494,    '0.975' => 47.924,   '0.200' => 78.643,   '0.100' => 84.418,   '0.050' => 89.391,   '0.025' => 93.856,   '0.020' => 95.213,   '0.010' => 99.228,   '0.005' => 102.996,  '0.002' => 107.685,  '0.001' => 111.055],
        70   => ['0.995' => 43.275,    '0.975' => 48.758,   '0.200' => 79.715,   '0.100' => 85.527,   '0.050' => 90.531,   '0.025' => 95.023,   '0.020' => 96.388,   '0.010' => 100.425,  '0.005' => 104.215,  '0.002' => 108.929,  '0.001' => 112.317],
        71   => ['0.995' => 44.058,    '0.975' => 49.592,   '0.200' => 80.786,   '0.100' => 86.635,   '0.050' => 91.670,   '0.025' => 96.189,   '0.020' => 97.561,   '0.010' => 101.621,  '0.005' => 105.432,  '0.002' => 110.172,  '0.001' => 113.577],
        72   => ['0.995' => 44.843,    '0.975' => 50.428,   '0.200' => 81.857,   '0.100' => 87.743,   '0.050' => 92.808,   '0.025' => 97.353,   '0.020' => 98.733,   '0.010' => 102.816,  '0.005' => 106.648,  '0.002' => 111.412,  '0.001' => 114.835],
        73   => ['0.995' => 45.629,    '0.975' => 51.265,   '0.200' => 82.927,   '0.100' => 88.850,   '0.050' => 93.945,   '0.025' => 98.516,   '0.020' => 99.904,   '0.010' => 104.010,  '0.005' => 107.862,  '0.002' => 112.651,  '0.001' => 116.092],
        74   => ['0.995' => 46.417,    '0.975' => 52.103,   '0.200' => 83.997,   '0.100' => 89.956,   '0.050' => 95.081,   '0.025' => 99.678,   '0.020' => 101.074,  '0.010' => 105.202,  '0.005' => 109.074,  '0.002' => 113.889,  '0.001' => 117.346],
        75   => ['0.995' => 47.206,    '0.975' => 52.942,   '0.200' => 85.066,   '0.100' => 91.061,   '0.050' => 96.217,   '0.025' => 100.839,  '0.020' => 102.243,  '0.010' => 106.393,  '0.005' => 110.286,  '0.002' => 115.125,  '0.001' => 118.599],
        76   => ['0.995' => 47.997,    '0.975' => 53.782,   '0.200' => 86.135,   '0.100' => 92.166,   '0.050' => 97.351,   '0.025' => 101.999,  '0.020' => 103.410,  '0.010' => 107.583,  '0.005' => 111.495,  '0.002' => 116.359,  '0.001' => 119.850],
        77   => ['0.995' => 48.788,    '0.975' => 54.623,   '0.200' => 87.203,   '0.100' => 93.270,   '0.050' => 98.484,   '0.025' => 103.158,  '0.020' => 104.576,  '0.010' => 108.771,  '0.005' => 112.704,  '0.002' => 117.591,  '0.001' => 121.100],
        78   => ['0.995' => 49.582,    '0.975' => 55.466,   '0.200' => 88.271,   '0.100' => 94.374,   '0.050' => 99.617,   '0.025' => 104.316,  '0.020' => 105.742,  '0.010' => 109.958,  '0.005' => 113.911,  '0.002' => 118.823,  '0.001' => 122.348],
        79   => ['0.995' => 50.376,    '0.975' => 56.309,   '0.200' => 89.338,   '0.100' => 95.476,   '0.050' => 100.749,  '0.025' => 105.473,  '0.020' => 106.906,  '0.010' => 111.144,  '0.005' => 115.117,  '0.002' => 120.052,  '0.001' => 123.594],
        80   => ['0.995' => 51.172,    '0.975' => 57.153,   '0.200' => 90.405,   '0.100' => 96.578,   '0.050' => 101.879,  '0.025' => 106.629,  '0.020' => 108.069,  '0.010' => 112.329,  '0.005' => 116.321,  '0.002' => 121.280,  '0.001' => 124.839],
        81   => ['0.995' => 51.969,    '0.975' => 57.998,   '0.200' => 91.472,   '0.100' => 97.680,   '0.050' => 103.010,  '0.025' => 107.783,  '0.020' => 109.232,  '0.010' => 113.512,  '0.005' => 117.524,  '0.002' => 122.507,  '0.001' => 126.083],
        82   => ['0.995' => 52.767,    '0.975' => 58.845,   '0.200' => 92.538,   '0.100' => 98.780,   '0.050' => 104.139,  '0.025' => 108.937,  '0.020' => 110.393,  '0.010' => 114.695,  '0.005' => 118.726,  '0.002' => 123.733,  '0.001' => 127.324],
        83   => ['0.995' => 53.567,    '0.975' => 59.692,   '0.200' => 93.604,   '0.100' => 99.880,   '0.050' => 105.267,  '0.025' => 110.090,  '0.020' => 111.553,  '0.010' => 115.876,  '0.005' => 119.927,  '0.002' => 124.957,  '0.001' => 128.565],
        84   => ['0.995' => 54.368,    '0.975' => 60.540,   '0.200' => 94.669,   '0.100' => 100.980,  '0.050' => 106.395,  '0.025' => 111.242,  '0.020' => 112.712,  '0.010' => 117.057,  '0.005' => 121.126,  '0.002' => 126.179,  '0.001' => 129.804],
        85   => ['0.995' => 55.170,    '0.975' => 61.389,   '0.200' => 95.734,   '0.100' => 102.079,  '0.050' => 107.522,  '0.025' => 112.393,  '0.020' => 113.871,  '0.010' => 118.236,  '0.005' => 122.325,  '0.002' => 127.401,  '0.001' => 131.041],
        86   => ['0.995' => 55.973,    '0.975' => 62.239,   '0.200' => 96.799,   '0.100' => 103.177,  '0.050' => 108.648,  '0.025' => 113.544,  '0.020' => 115.028,  '0.010' => 119.414,  '0.005' => 123.522,  '0.002' => 128.621,  '0.001' => 132.277],
        87   => ['0.995' => 56.777,    '0.975' => 63.089,   '0.200' => 97.863,   '0.100' => 104.275,  '0.050' => 109.773,  '0.025' => 114.693,  '0.020' => 116.184,  '0.010' => 120.591,  '0.005' => 124.718,  '0.002' => 129.840,  '0.001' => 133.512],
        88   => ['0.995' => 57.582,    '0.975' => 63.941,   '0.200' => 98.927,   '0.100' => 105.372,  '0.050' => 110.898,  '0.025' => 115.841,  '0.020' => 117.340,  '0.010' => 121.767,  '0.005' => 125.913,  '0.002' => 131.057,  '0.001' => 134.745],
        89   => ['0.995' => 58.389,    '0.975' => 64.793,   '0.200' => 99.991,   '0.100' => 106.469,  '0.050' => 112.022,  '0.025' => 116.989,  '0.020' => 118.495,  '0.010' => 122.942,  '0.005' => 127.106,  '0.002' => 132.273,  '0.001' => 135.978],
        90   => ['0.995' => 59.196,    '0.975' => 65.647,   '0.200' => 101.054,  '0.100' => 107.565,  '0.050' => 113.145,  '0.025' => 118.136,  '0.020' => 119.648,  '0.010' => 124.116,  '0.005' => 128.299,  '0.002' => 133.489,  '0.001' => 137.208],
        91   => ['0.995' => 60.005,    '0.975' => 66.501,   '0.200' => 102.117,  '0.100' => 108.661,  '0.050' => 114.268,  '0.025' => 119.282,  '0.020' => 120.801,  '0.010' => 125.289,  '0.005' => 129.491,  '0.002' => 134.702,  '0.001' => 138.438],
        92   => ['0.995' => 60.815,    '0.975' => 67.356,   '0.200' => 103.179,  '0.100' => 109.756,  '0.050' => 115.390,  '0.025' => 120.427,  '0.020' => 121.954,  '0.010' => 126.462,  '0.005' => 130.681,  '0.002' => 135.915,  '0.001' => 139.666],
        93   => ['0.995' => 61.625,    '0.975' => 68.211,   '0.200' => 104.241,  '0.100' => 110.850,  '0.050' => 116.511,  '0.025' => 121.571,  '0.020' => 123.105,  '0.010' => 127.633,  '0.005' => 131.871,  '0.002' => 137.127,  '0.001' => 140.893],
        94   => ['0.995' => 62.437,    '0.975' => 69.068,   '0.200' => 105.303,  '0.100' => 111.944,  '0.050' => 117.632,  '0.025' => 122.715,  '0.020' => 124.255,  '0.010' => 128.803,  '0.005' => 133.059,  '0.002' => 138.337,  '0.001' => 142.119],
        95   => ['0.995' => 63.250,    '0.975' => 69.925,   '0.200' => 106.364,  '0.100' => 113.038,  '0.050' => 118.752,  '0.025' => 123.858,  '0.020' => 125.405,  '0.010' => 129.973,  '0.005' => 134.247,  '0.002' => 139.546,  '0.001' => 143.344],
        96   => ['0.995' => 64.063,    '0.975' => 70.783,   '0.200' => 107.425,  '0.100' => 114.131,  '0.050' => 119.871,  '0.025' => 125.000,  '0.020' => 126.554,  '0.010' => 131.141,  '0.005' => 135.433,  '0.002' => 140.755,  '0.001' => 144.567],
        97   => ['0.995' => 64.878,    '0.975' => 71.642,   '0.200' => 108.486,  '0.100' => 115.223,  '0.050' => 120.990,  '0.025' => 126.141,  '0.020' => 127.702,  '0.010' => 132.309,  '0.005' => 136.619,  '0.002' => 141.962,  '0.001' => 145.789],
        98   => ['0.995' => 65.694,    '0.975' => 72.501,   '0.200' => 109.547,  '0.100' => 116.315,  '0.050' => 122.108,  '0.025' => 127.282,  '0.020' => 128.849,  '0.010' => 133.476,  '0.005' => 137.803,  '0.002' => 143.168,  '0.001' => 147.010],
        99   => ['0.995' => 66.510,    '0.975' => 73.361,   '0.200' => 110.607,  '0.100' => 117.407,  '0.050' => 123.225,  '0.025' => 128.422,  '0.020' => 129.996,  '0.010' => 134.642,  '0.005' => 138.987,  '0.002' => 144.373,  '0.001' => 148.230],
        100  => ['0.995' => 67.328,    '0.975' => 74.222,   '0.200' => 111.667,  '0.100' => 118.498,  '0.050' => 124.342,  '0.025' => 129.561,  '0.020' => 131.142,  '0.010' => 135.807,  '0.005' => 140.169,  '0.002' => 145.577,  '0.001' => 149.449],
        101  => ['0.995' => 68.146,    '0.975' => 75.083,   '0.200' => 112.726,  '0.100' => 119.589,  '0.050' => 125.458,  '0.025' => 130.700,  '0.020' => 132.287,  '0.010' => 136.971,  '0.005' => 141.351,  '0.002' => 146.780,  '0.001' => 150.667],
        102  => ['0.995' => 68.965,    '0.975' => 75.946,   '0.200' => 113.786,  '0.100' => 120.679,  '0.050' => 126.574,  '0.025' => 131.838,  '0.020' => 133.431,  '0.010' => 138.134,  '0.005' => 142.532,  '0.002' => 147.982,  '0.001' => 151.884],
        103  => ['0.995' => 69.785,    '0.975' => 76.809,   '0.200' => 114.845,  '0.100' => 121.769,  '0.050' => 127.689,  '0.025' => 132.975,  '0.020' => 134.575,  '0.010' => 139.297,  '0.005' => 143.712,  '0.002' => 149.183,  '0.001' => 153.099],
        104  => ['0.995' => 70.606,    '0.975' => 77.672,   '0.200' => 115.903,  '0.100' => 122.858,  '0.050' => 128.804,  '0.025' => 134.111,  '0.020' => 135.718,  '0.010' => 140.459,  '0.005' => 144.891,  '0.002' => 150.383,  '0.001' => 154.314],
        105  => ['0.995' => 71.428,    '0.975' => 78.536,   '0.200' => 116.962,  '0.100' => 123.947,  '0.050' => 129.918,  '0.025' => 135.247,  '0.020' => 136.860,  '0.010' => 141.620,  '0.005' => 146.070,  '0.002' => 151.582,  '0.001' => 155.528],
        106  => ['0.995' => 72.251,    '0.975' => 79.401,   '0.200' => 118.020,  '0.100' => 125.035,  '0.050' => 131.031,  '0.025' => 136.382,  '0.020' => 138.002,  '0.010' => 142.780,  '0.005' => 147.247,  '0.002' => 152.780,  '0.001' => 156.740],
        107  => ['0.995' => 73.075,    '0.975' => 80.267,   '0.200' => 119.078,  '0.100' => 126.123,  '0.050' => 132.144,  '0.025' => 137.517,  '0.020' => 139.143,  '0.010' => 143.940,  '0.005' => 148.424,  '0.002' => 153.977,  '0.001' => 157.952],
        108  => ['0.995' => 73.899,    '0.975' => 81.133,   '0.200' => 120.135,  '0.100' => 127.211,  '0.050' => 133.257,  '0.025' => 138.651,  '0.020' => 140.283,  '0.010' => 145.099,  '0.005' => 149.599,  '0.002' => 155.173,  '0.001' => 159.162],
        109  => ['0.995' => 74.724,    '0.975' => 82.000,   '0.200' => 121.192,  '0.100' => 128.298,  '0.050' => 134.369,  '0.025' => 139.784,  '0.020' => 141.423,  '0.010' => 146.257,  '0.005' => 150.774,  '0.002' => 156.369,  '0.001' => 160.372],
        110  => ['0.995' => 75.550,    '0.975' => 82.867,   '0.200' => 122.250,  '0.100' => 129.385,  '0.050' => 135.480,  '0.025' => 140.917,  '0.020' => 142.562,  '0.010' => 147.414,  '0.005' => 151.948,  '0.002' => 157.563,  '0.001' => 161.581],
        111  => ['0.995' => 76.377,    '0.975' => 83.735,   '0.200' => 123.306,  '0.100' => 130.472,  '0.050' => 136.591,  '0.025' => 142.049,  '0.020' => 143.700,  '0.010' => 148.571,  '0.005' => 153.122,  '0.002' => 158.757,  '0.001' => 162.788],
        112  => ['0.995' => 77.204,    '0.975' => 84.604,   '0.200' => 124.363,  '0.100' => 131.558,  '0.050' => 137.701,  '0.025' => 143.180,  '0.020' => 144.838,  '0.010' => 149.727,  '0.005' => 154.294,  '0.002' => 159.950,  '0.001' => 163.995],
        113  => ['0.995' => 78.033,    '0.975' => 85.473,   '0.200' => 125.419,  '0.100' => 132.643,  '0.050' => 138.811,  '0.025' => 144.311,  '0.020' => 145.975,  '0.010' => 150.882,  '0.005' => 155.466,  '0.002' => 161.141,  '0.001' => 165.201],
        114  => ['0.995' => 78.862,    '0.975' => 86.342,   '0.200' => 126.475,  '0.100' => 133.729,  '0.050' => 139.921,  '0.025' => 145.441,  '0.020' => 147.111,  '0.010' => 152.037,  '0.005' => 156.637,  '0.002' => 162.332,  '0.001' => 166.406],
        115  => ['0.995' => 79.692,    '0.975' => 87.213,   '0.200' => 127.531,  '0.100' => 134.813,  '0.050' => 141.030,  '0.025' => 146.571,  '0.020' => 148.247,  '0.010' => 153.191,  '0.005' => 157.808,  '0.002' => 163.523,  '0.001' => 167.610],
        116  => ['0.995' => 80.522,    '0.975' => 88.084,   '0.200' => 128.587,  '0.100' => 135.898,  '0.050' => 142.138,  '0.025' => 147.700,  '0.020' => 149.383,  '0.010' => 154.344,  '0.005' => 158.977,  '0.002' => 164.712,  '0.001' => 168.813],
        117  => ['0.995' => 81.353,    '0.975' => 88.955,   '0.200' => 129.642,  '0.100' => 136.982,  '0.050' => 143.246,  '0.025' => 148.829,  '0.020' => 150.517,  '0.010' => 155.496,  '0.005' => 160.146,  '0.002' => 165.900,  '0.001' => 170.016],
        118  => ['0.995' => 82.185,    '0.975' => 89.827,   '0.200' => 130.697,  '0.100' => 138.066,  '0.050' => 144.354,  '0.025' => 149.957,  '0.020' => 151.652,  '0.010' => 156.648,  '0.005' => 161.314,  '0.002' => 167.088,  '0.001' => 171.217],
        119  => ['0.995' => 83.018,    '0.975' => 90.700,   '0.200' => 131.752,  '0.100' => 139.149,  '0.050' => 145.461,  '0.025' => 151.084,  '0.020' => 152.785,  '0.010' => 157.800,  '0.005' => 162.481,  '0.002' => 168.275,  '0.001' => 172.418],
        120  => ['0.995' => 83.852,    '0.975' => 91.573,   '0.200' => 132.806,  '0.100' => 140.233,  '0.050' => 146.567,  '0.025' => 152.211,  '0.020' => 153.918,  '0.010' => 158.950,  '0.005' => 163.648,  '0.002' => 169.461,  '0.001' => 173.617],
        121  => ['0.995' => 84.686,    '0.975' => 92.446,   '0.200' => 133.861,  '0.100' => 141.315,  '0.050' => 147.674,  '0.025' => 153.338,  '0.020' => 155.051,  '0.010' => 160.100,  '0.005' => 164.814,  '0.002' => 170.647,  '0.001' => 174.816],
        122  => ['0.995' => 85.520,    '0.975' => 93.320,   '0.200' => 134.915,  '0.100' => 142.398,  '0.050' => 148.779,  '0.025' => 154.464,  '0.020' => 156.183,  '0.010' => 161.250,  '0.005' => 165.980,  '0.002' => 171.831,  '0.001' => 176.014],
        123  => ['0.995' => 86.356,    '0.975' => 94.195,   '0.200' => 135.969,  '0.100' => 143.480,  '0.050' => 149.885,  '0.025' => 155.589,  '0.020' => 157.314,  '0.010' => 162.398,  '0.005' => 167.144,  '0.002' => 173.015,  '0.001' => 177.212],
        124  => ['0.995' => 87.192,    '0.975' => 95.070,   '0.200' => 137.022,  '0.100' => 144.562,  '0.050' => 150.989,  '0.025' => 156.714,  '0.020' => 158.445,  '0.010' => 163.546,  '0.005' => 168.308,  '0.002' => 174.198,  '0.001' => 178.408],
        125  => ['0.995' => 88.029,    '0.975' => 95.946,   '0.200' => 138.076,  '0.100' => 145.643,  '0.050' => 152.094,  '0.025' => 157.839,  '0.020' => 159.575,  '0.010' => 164.694,  '0.005' => 169.471,  '0.002' => 175.380,  '0.001' => 179.604],
        126  => ['0.995' => 88.866,    '0.975' => 96.822,   '0.200' => 139.129,  '0.100' => 146.724,  '0.050' => 153.198,  '0.025' => 158.962,  '0.020' => 160.705,  '0.010' => 165.841,  '0.005' => 170.634,  '0.002' => 176.562,  '0.001' => 180.799],
        127  => ['0.995' => 89.704,    '0.975' => 97.698,   '0.200' => 140.182,  '0.100' => 147.805,  '0.050' => 154.302,  '0.025' => 160.086,  '0.020' => 161.834,  '0.010' => 166.987,  '0.005' => 171.796,  '0.002' => 177.743,  '0.001' => 181.993],
        128  => ['0.995' => 90.543,    '0.975' => 98.576,   '0.200' => 141.235,  '0.100' => 148.885,  '0.050' => 155.405,  '0.025' => 161.209,  '0.020' => 162.963,  '0.010' => 168.133,  '0.005' => 172.957,  '0.002' => 178.923,  '0.001' => 183.186],
        129  => ['0.995' => 91.382,    '0.975' => 99.453,   '0.200' => 142.288,  '0.100' => 149.965,  '0.050' => 156.508,  '0.025' => 162.331,  '0.020' => 164.091,  '0.010' => 169.278,  '0.005' => 174.118,  '0.002' => 180.103,  '0.001' => 184.379],
        130  => ['0.995' => 92.222,    '0.975' => 100.331,  '0.200' => 143.340,  '0.100' => 151.045,  '0.050' => 157.610,  '0.025' => 163.453,  '0.020' => 165.219,  '0.010' => 170.423,  '0.005' => 175.278,  '0.002' => 181.282,  '0.001' => 185.571],
        131  => ['0.995' => 93.063,    '0.975' => 101.210,  '0.200' => 144.392,  '0.100' => 152.125,  '0.050' => 158.712,  '0.025' => 164.575,  '0.020' => 166.346,  '0.010' => 171.567,  '0.005' => 176.438,  '0.002' => 182.460,  '0.001' => 186.762],
        132  => ['0.995' => 93.904,    '0.975' => 102.089,  '0.200' => 145.444,  '0.100' => 153.204,  '0.050' => 159.814,  '0.025' => 165.696,  '0.020' => 167.473,  '0.010' => 172.711,  '0.005' => 177.597,  '0.002' => 183.637,  '0.001' => 187.953],
        133  => ['0.995' => 94.746,    '0.975' => 102.968,  '0.200' => 146.496,  '0.100' => 154.283,  '0.050' => 160.915,  '0.025' => 166.816,  '0.020' => 168.600,  '0.010' => 173.854,  '0.005' => 178.755,  '0.002' => 184.814,  '0.001' => 189.142],
        134  => ['0.995' => 95.588,    '0.975' => 103.848,  '0.200' => 147.548,  '0.100' => 155.361,  '0.050' => 162.016,  '0.025' => 167.936,  '0.020' => 169.725,  '0.010' => 174.996,  '0.005' => 179.913,  '0.002' => 185.990,  '0.001' => 190.331],
        135  => ['0.995' => 96.431,    '0.975' => 104.729,  '0.200' => 148.599,  '0.100' => 156.440,  '0.050' => 163.116,  '0.025' => 169.056,  '0.020' => 170.851,  '0.010' => 176.138,  '0.005' => 181.070,  '0.002' => 187.165,  '0.001' => 191.520],
        136  => ['0.995' => 97.275,    '0.975' => 105.609,  '0.200' => 149.651,  '0.100' => 157.518,  '0.050' => 164.216,  '0.025' => 170.175,  '0.020' => 171.976,  '0.010' => 177.280,  '0.005' => 182.226,  '0.002' => 188.340,  '0.001' => 192.707],
        137  => ['0.995' => 98.119,    '0.975' => 106.491,  '0.200' => 150.702,  '0.100' => 158.595,  '0.050' => 165.316,  '0.025' => 171.294,  '0.020' => 173.100,  '0.010' => 178.421,  '0.005' => 183.382,  '0.002' => 189.514,  '0.001' => 193.894],
        138  => ['0.995' => 98.964,    '0.975' => 107.372,  '0.200' => 151.753,  '0.100' => 159.673,  '0.050' => 166.415,  '0.025' => 172.412,  '0.020' => 174.224,  '0.010' => 179.561,  '0.005' => 184.538,  '0.002' => 190.688,  '0.001' => 195.080],
        139  => ['0.995' => 99.809,    '0.975' => 108.254,  '0.200' => 152.803,  '0.100' => 160.750,  '0.050' => 167.514,  '0.025' => 173.530,  '0.020' => 175.348,  '0.010' => 180.701,  '0.005' => 185.693,  '0.002' => 191.861,  '0.001' => 196.266],
        140  => ['0.995' => 100.655,   '0.975' => 109.137,  '0.200' => 153.854,  '0.100' => 161.827,  '0.050' => 168.613,  '0.025' => 174.648,  '0.020' => 176.471,  '0.010' => 181.840,  '0.005' => 186.847,  '0.002' => 193.033,  '0.001' => 197.451],
        141  => ['0.995' => 101.501,   '0.975' => 110.020,  '0.200' => 154.904,  '0.100' => 162.904,  '0.050' => 169.711,  '0.025' => 175.765,  '0.020' => 177.594,  '0.010' => 182.979,  '0.005' => 188.001,  '0.002' => 194.205,  '0.001' => 198.635],
        142  => ['0.995' => 102.348,   '0.975' => 110.903,  '0.200' => 155.954,  '0.100' => 163.980,  '0.050' => 170.809,  '0.025' => 176.882,  '0.020' => 178.716,  '0.010' => 184.118,  '0.005' => 189.154,  '0.002' => 195.376,  '0.001' => 199.819],
        143  => ['0.995' => 103.196,   '0.975' => 111.787,  '0.200' => 157.004,  '0.100' => 165.056,  '0.050' => 171.907,  '0.025' => 177.998,  '0.020' => 179.838,  '0.010' => 185.256,  '0.005' => 190.306,  '0.002' => 196.546,  '0.001' => 201.002],
        144  => ['0.995' => 104.044,   '0.975' => 112.671,  '0.200' => 158.054,  '0.100' => 166.132,  '0.050' => 173.004,  '0.025' => 179.114,  '0.020' => 180.959,  '0.010' => 186.393,  '0.005' => 191.458,  '0.002' => 197.716,  '0.001' => 202.184],
        145  => ['0.995' => 104.892,   '0.975' => 113.556,  '0.200' => 159.104,  '0.100' => 167.207,  '0.050' => 174.101,  '0.025' => 180.229,  '0.020' => 182.080,  '0.010' => 187.530,  '0.005' => 192.610,  '0.002' => 198.885,  '0.001' => 203.366],
        146  => ['0.995' => 105.741,   '0.975' => 114.441,  '0.200' => 160.153,  '0.100' => 168.283,  '0.050' => 175.198,  '0.025' => 181.344,  '0.020' => 183.200,  '0.010' => 188.666,  '0.005' => 193.761,  '0.002' => 200.054,  '0.001' => 204.547],
        147  => ['0.995' => 106.591,   '0.975' => 115.326,  '0.200' => 161.202,  '0.100' => 169.358,  '0.050' => 176.294,  '0.025' => 182.459,  '0.020' => 184.321,  '0.010' => 189.802,  '0.005' => 194.912,  '0.002' => 201.222,  '0.001' => 205.727],
        148  => ['0.995' => 107.441,   '0.975' => 116.212,  '0.200' => 162.251,  '0.100' => 170.432,  '0.050' => 177.390,  '0.025' => 183.573,  '0.020' => 185.440,  '0.010' => 190.938,  '0.005' => 196.062,  '0.002' => 202.390,  '0.001' => 206.907],
        149  => ['0.995' => 108.291,   '0.975' => 117.098,  '0.200' => 163.300,  '0.100' => 171.507,  '0.050' => 178.485,  '0.025' => 184.687,  '0.020' => 186.560,  '0.010' => 192.073,  '0.005' => 197.211,  '0.002' => 203.557,  '0.001' => 208.086],
        150  => ['0.995' => 109.142,   '0.975' => 117.985,  '0.200' => 164.349,  '0.100' => 172.581,  '0.050' => 179.581,  '0.025' => 185.800,  '0.020' => 187.678,  '0.010' => 193.208,  '0.005' => 198.360,  '0.002' => 204.723,  '0.001' => 209.265],
        151  => ['0.995' => 109.994,   '0.975' => 118.871,  '0.200' => 165.398,  '0.100' => 173.655,  '0.050' => 180.676,  '0.025' => 186.914,  '0.020' => 188.797,  '0.010' => 194.342,  '0.005' => 199.509,  '0.002' => 205.889,  '0.001' => 210.443],
        152  => ['0.995' => 110.846,   '0.975' => 119.759,  '0.200' => 166.446,  '0.100' => 174.729,  '0.050' => 181.770,  '0.025' => 188.026,  '0.020' => 189.915,  '0.010' => 195.476,  '0.005' => 200.657,  '0.002' => 207.054,  '0.001' => 211.620],
        153  => ['0.995' => 111.698,   '0.975' => 120.646,  '0.200' => 167.495,  '0.100' => 175.803,  '0.050' => 182.865,  '0.025' => 189.139,  '0.020' => 191.033,  '0.010' => 196.609,  '0.005' => 201.804,  '0.002' => 208.219,  '0.001' => 212.797],
        154  => ['0.995' => 112.551,   '0.975' => 121.534,  '0.200' => 168.543,  '0.100' => 176.876,  '0.050' => 183.959,  '0.025' => 190.251,  '0.020' => 192.150,  '0.010' => 197.742,  '0.005' => 202.951,  '0.002' => 209.383,  '0.001' => 213.973],
        155  => ['0.995' => 113.405,   '0.975' => 122.423,  '0.200' => 169.591,  '0.100' => 177.949,  '0.050' => 185.052,  '0.025' => 191.362,  '0.020' => 193.267,  '0.010' => 198.874,  '0.005' => 204.098,  '0.002' => 210.547,  '0.001' => 215.149],
        156  => ['0.995' => 114.259,   '0.975' => 123.312,  '0.200' => 170.639,  '0.100' => 179.022,  '0.050' => 186.146,  '0.025' => 192.474,  '0.020' => 194.384,  '0.010' => 200.006,  '0.005' => 205.244,  '0.002' => 211.710,  '0.001' => 216.324],
        157  => ['0.995' => 115.113,   '0.975' => 124.201,  '0.200' => 171.686,  '0.100' => 180.094,  '0.050' => 187.239,  '0.025' => 193.584,  '0.020' => 195.500,  '0.010' => 201.138,  '0.005' => 206.390,  '0.002' => 212.873,  '0.001' => 217.499],
        158  => ['0.995' => 115.968,   '0.975' => 125.090,  '0.200' => 172.734,  '0.100' => 181.167,  '0.050' => 188.332,  '0.025' => 194.695,  '0.020' => 196.616,  '0.010' => 202.269,  '0.005' => 207.535,  '0.002' => 214.035,  '0.001' => 218.673],
        159  => ['0.995' => 116.823,   '0.975' => 125.980,  '0.200' => 173.781,  '0.100' => 182.239,  '0.050' => 189.424,  '0.025' => 195.805,  '0.020' => 197.731,  '0.010' => 203.400,  '0.005' => 208.680,  '0.002' => 215.197,  '0.001' => 219.846],
        160  => ['0.995' => 117.679,   '0.975' => 126.870,  '0.200' => 174.828,  '0.100' => 183.311,  '0.050' => 190.516,  '0.025' => 196.915,  '0.020' => 198.846,  '0.010' => 204.530,  '0.005' => 209.824,  '0.002' => 216.358,  '0.001' => 221.019],
        161  => ['0.995' => 118.536,   '0.975' => 127.761,  '0.200' => 175.875,  '0.100' => 184.382,  '0.050' => 191.608,  '0.025' => 198.025,  '0.020' => 199.961,  '0.010' => 205.660,  '0.005' => 210.968,  '0.002' => 217.518,  '0.001' => 222.191],
        162  => ['0.995' => 119.392,   '0.975' => 128.651,  '0.200' => 176.922,  '0.100' => 185.454,  '0.050' => 192.700,  '0.025' => 199.134,  '0.020' => 201.076,  '0.010' => 206.790,  '0.005' => 212.111,  '0.002' => 218.678,  '0.001' => 223.363],
        163  => ['0.995' => 120.249,   '0.975' => 129.543,  '0.200' => 177.969,  '0.100' => 186.525,  '0.050' => 193.791,  '0.025' => 200.243,  '0.020' => 202.190,  '0.010' => 207.919,  '0.005' => 213.254,  '0.002' => 219.838,  '0.001' => 224.535],
        164  => ['0.995' => 121.107,   '0.975' => 130.434,  '0.200' => 179.016,  '0.100' => 187.596,  '0.050' => 194.883,  '0.025' => 201.351,  '0.020' => 203.303,  '0.010' => 209.047,  '0.005' => 214.396,  '0.002' => 220.997,  '0.001' => 225.705],
        165  => ['0.995' => 121.965,   '0.975' => 131.326,  '0.200' => 180.062,  '0.100' => 188.667,  '0.050' => 195.973,  '0.025' => 202.459,  '0.020' => 204.417,  '0.010' => 210.176,  '0.005' => 215.539,  '0.002' => 222.156,  '0.001' => 226.876],
        166  => ['0.995' => 122.823,   '0.975' => 132.218,  '0.200' => 181.109,  '0.100' => 189.737,  '0.050' => 197.064,  '0.025' => 203.567,  '0.020' => 205.530,  '0.010' => 211.304,  '0.005' => 216.680,  '0.002' => 223.314,  '0.001' => 228.045],
        167  => ['0.995' => 123.682,   '0.975' => 133.111,  '0.200' => 182.155,  '0.100' => 190.808,  '0.050' => 198.154,  '0.025' => 204.675,  '0.020' => 206.642,  '0.010' => 212.431,  '0.005' => 217.821,  '0.002' => 224.472,  '0.001' => 229.215],
        168  => ['0.995' => 124.541,   '0.975' => 134.003,  '0.200' => 183.201,  '0.100' => 191.878,  '0.050' => 199.244,  '0.025' => 205.782,  '0.020' => 207.755,  '0.010' => 213.558,  '0.005' => 218.962,  '0.002' => 225.629,  '0.001' => 230.383],
        169  => ['0.995' => 125.401,   '0.975' => 134.897,  '0.200' => 184.247,  '0.100' => 192.948,  '0.050' => 200.334,  '0.025' => 206.889,  '0.020' => 208.867,  '0.010' => 214.685,  '0.005' => 220.102,  '0.002' => 226.786,  '0.001' => 231.552],
        170  => ['0.995' => 126.261,   '0.975' => 135.790,  '0.200' => 185.293,  '0.100' => 194.017,  '0.050' => 201.423,  '0.025' => 207.995,  '0.020' => 209.978,  '0.010' => 215.812,  '0.005' => 221.242,  '0.002' => 227.942,  '0.001' => 232.719],
        171  => ['0.995' => 127.122,   '0.975' => 136.684,  '0.200' => 186.338,  '0.100' => 195.087,  '0.050' => 202.513,  '0.025' => 209.102,  '0.020' => 211.090,  '0.010' => 216.938,  '0.005' => 222.382,  '0.002' => 229.098,  '0.001' => 233.887],
        172  => ['0.995' => 127.983,   '0.975' => 137.578,  '0.200' => 187.384,  '0.100' => 196.156,  '0.050' => 203.602,  '0.025' => 210.208,  '0.020' => 212.201,  '0.010' => 218.063,  '0.005' => 223.521,  '0.002' => 230.253,  '0.001' => 235.053],
        173  => ['0.995' => 128.844,   '0.975' => 138.472,  '0.200' => 188.429,  '0.100' => 197.225,  '0.050' => 204.690,  '0.025' => 211.313,  '0.020' => 213.311,  '0.010' => 219.189,  '0.005' => 224.660,  '0.002' => 231.408,  '0.001' => 236.220],
        174  => ['0.995' => 129.706,   '0.975' => 139.367,  '0.200' => 189.475,  '0.100' => 198.294,  '0.050' => 205.779,  '0.025' => 212.419,  '0.020' => 214.422,  '0.010' => 220.314,  '0.005' => 225.798,  '0.002' => 232.563,  '0.001' => 237.385],
        175  => ['0.995' => 130.568,   '0.975' => 140.262,  '0.200' => 190.520,  '0.100' => 199.363,  '0.050' => 206.867,  '0.025' => 213.524,  '0.020' => 215.532,  '0.010' => 221.438,  '0.005' => 226.936,  '0.002' => 233.717,  '0.001' => 238.551],
        176  => ['0.995' => 131.430,   '0.975' => 141.157,  '0.200' => 191.565,  '0.100' => 200.432,  '0.050' => 207.955,  '0.025' => 214.628,  '0.020' => 216.641,  '0.010' => 222.563,  '0.005' => 228.074,  '0.002' => 234.870,  '0.001' => 239.716],
        177  => ['0.995' => 132.293,   '0.975' => 142.053,  '0.200' => 192.610,  '0.100' => 201.500,  '0.050' => 209.042,  '0.025' => 215.733,  '0.020' => 217.751,  '0.010' => 223.687,  '0.005' => 229.211,  '0.002' => 236.023,  '0.001' => 240.880],
        178  => ['0.995' => 133.157,   '0.975' => 142.949,  '0.200' => 193.654,  '0.100' => 202.568,  '0.050' => 210.130,  '0.025' => 216.837,  '0.020' => 218.860,  '0.010' => 224.810,  '0.005' => 230.347,  '0.002' => 237.176,  '0.001' => 242.044],
        179  => ['0.995' => 134.020,   '0.975' => 143.845,  '0.200' => 194.699,  '0.100' => 203.636,  '0.050' => 211.217,  '0.025' => 217.941,  '0.020' => 219.969,  '0.010' => 225.933,  '0.005' => 231.484,  '0.002' => 238.328,  '0.001' => 243.207],
        180  => ['0.995' => 134.884,   '0.975' => 144.741,  '0.200' => 195.743,  '0.100' => 204.704,  '0.050' => 212.304,  '0.025' => 219.044,  '0.020' => 221.077,  '0.010' => 227.056,  '0.005' => 232.620,  '0.002' => 239.480,  '0.001' => 244.370],
        181  => ['0.995' => 135.749,   '0.975' => 145.638,  '0.200' => 196.788,  '0.100' => 205.771,  '0.050' => 213.391,  '0.025' => 220.148,  '0.020' => 222.185,  '0.010' => 228.179,  '0.005' => 233.755,  '0.002' => 240.632,  '0.001' => 245.533],
        182  => ['0.995' => 136.614,   '0.975' => 146.535,  '0.200' => 197.832,  '0.100' => 206.839,  '0.050' => 214.477,  '0.025' => 221.251,  '0.020' => 223.293,  '0.010' => 229.301,  '0.005' => 234.891,  '0.002' => 241.783,  '0.001' => 246.695],
        183  => ['0.995' => 137.479,   '0.975' => 147.432,  '0.200' => 198.876,  '0.100' => 207.906,  '0.050' => 215.563,  '0.025' => 222.353,  '0.020' => 224.401,  '0.010' => 230.423,  '0.005' => 236.026,  '0.002' => 242.933,  '0.001' => 247.857],
        184  => ['0.995' => 138.344,   '0.975' => 148.330,  '0.200' => 199.920,  '0.100' => 208.973,  '0.050' => 216.649,  '0.025' => 223.456,  '0.020' => 225.508,  '0.010' => 231.544,  '0.005' => 237.160,  '0.002' => 244.084,  '0.001' => 249.018],
        185  => ['0.995' => 139.210,   '0.975' => 149.228,  '0.200' => 200.964,  '0.100' => 210.040,  '0.050' => 217.735,  '0.025' => 224.558,  '0.020' => 226.615,  '0.010' => 232.665,  '0.005' => 238.294,  '0.002' => 245.234,  '0.001' => 250.179],
        186  => ['0.995' => 140.077,   '0.975' => 150.126,  '0.200' => 202.008,  '0.100' => 211.106,  '0.050' => 218.820,  '0.025' => 225.660,  '0.020' => 227.722,  '0.010' => 233.786,  '0.005' => 239.428,  '0.002' => 246.383,  '0.001' => 251.339],
        187  => ['0.995' => 140.943,   '0.975' => 151.024,  '0.200' => 203.052,  '0.100' => 212.173,  '0.050' => 219.906,  '0.025' => 226.761,  '0.020' => 228.828,  '0.010' => 234.907,  '0.005' => 240.561,  '0.002' => 247.532,  '0.001' => 252.499],
        188  => ['0.995' => 141.810,   '0.975' => 151.923,  '0.200' => 204.095,  '0.100' => 213.239,  '0.050' => 220.991,  '0.025' => 227.863,  '0.020' => 229.935,  '0.010' => 236.027,  '0.005' => 241.694,  '0.002' => 248.681,  '0.001' => 253.659],
        189  => ['0.995' => 142.678,   '0.975' => 152.822,  '0.200' => 205.139,  '0.100' => 214.305,  '0.050' => 222.076,  '0.025' => 228.964,  '0.020' => 231.040,  '0.010' => 237.147,  '0.005' => 242.827,  '0.002' => 249.829,  '0.001' => 254.818],
        190  => ['0.995' => 143.545,   '0.975' => 153.721,  '0.200' => 206.182,  '0.100' => 215.371,  '0.050' => 223.160,  '0.025' => 230.064,  '0.020' => 232.146,  '0.010' => 238.266,  '0.005' => 243.959,  '0.002' => 250.977,  '0.001' => 255.976],
        191  => ['0.995' => 144.413,   '0.975' => 154.621,  '0.200' => 207.225,  '0.100' => 216.437,  '0.050' => 224.245,  '0.025' => 231.165,  '0.020' => 233.251,  '0.010' => 239.386,  '0.005' => 245.091,  '0.002' => 252.124,  '0.001' => 257.135],
        192  => ['0.995' => 145.282,   '0.975' => 155.521,  '0.200' => 208.268,  '0.100' => 217.502,  '0.050' => 225.329,  '0.025' => 232.265,  '0.020' => 234.356,  '0.010' => 240.505,  '0.005' => 246.223,  '0.002' => 253.271,  '0.001' => 258.292],
        193  => ['0.995' => 146.150,   '0.975' => 156.421,  '0.200' => 209.311,  '0.100' => 218.568,  '0.050' => 226.413,  '0.025' => 233.365,  '0.020' => 235.461,  '0.010' => 241.623,  '0.005' => 247.354,  '0.002' => 254.418,  '0.001' => 259.450],
        194  => ['0.995' => 147.020,   '0.975' => 157.321,  '0.200' => 210.354,  '0.100' => 219.633,  '0.050' => 227.496,  '0.025' => 234.465,  '0.020' => 236.566,  '0.010' => 242.742,  '0.005' => 248.485,  '0.002' => 255.564,  '0.001' => 260.607],
        195  => ['0.995' => 147.889,   '0.975' => 158.221,  '0.200' => 211.397,  '0.100' => 220.698,  '0.050' => 228.580,  '0.025' => 235.564,  '0.020' => 237.670,  '0.010' => 243.860,  '0.005' => 249.616,  '0.002' => 256.710,  '0.001' => 261.763],
        196  => ['0.995' => 148.759,   '0.975' => 159.122,  '0.200' => 212.439,  '0.100' => 221.763,  '0.050' => 229.663,  '0.025' => 236.664,  '0.020' => 238.774,  '0.010' => 244.977,  '0.005' => 250.746,  '0.002' => 257.855,  '0.001' => 262.920],
        197  => ['0.995' => 149.629,   '0.975' => 160.023,  '0.200' => 213.482,  '0.100' => 222.828,  '0.050' => 230.746,  '0.025' => 237.763,  '0.020' => 239.877,  '0.010' => 246.095,  '0.005' => 251.876,  '0.002' => 259.001,  '0.001' => 264.075],
        198  => ['0.995' => 150.499,   '0.975' => 160.925,  '0.200' => 214.524,  '0.100' => 223.892,  '0.050' => 231.829,  '0.025' => 238.861,  '0.020' => 240.981,  '0.010' => 247.212,  '0.005' => 253.006,  '0.002' => 260.145,  '0.001' => 265.231],
        199  => ['0.995' => 151.370,   '0.975' => 161.826,  '0.200' => 215.567,  '0.100' => 224.957,  '0.050' => 232.912,  '0.025' => 239.960,  '0.020' => 242.084,  '0.010' => 248.329,  '0.005' => 254.135,  '0.002' => 261.290,  '0.001' => 266.386],
        200  => ['0.995' => 152.241,   '0.975' => 162.728,  '0.200' => 216.609,  '0.100' => 226.021,  '0.050' => 233.994,  '0.025' => 241.058,  '0.020' => 243.187,  '0.010' => 249.445,  '0.005' => 255.264,  '0.002' => 262.434,  '0.001' => 267.541],
        201  => ['0.995' => 153.112,   '0.975' => 163.630,  '0.200' => 217.651,  '0.100' => 227.085,  '0.050' => 235.077,  '0.025' => 242.156,  '0.020' => 244.290,  '0.010' => 250.561,  '0.005' => 256.393,  '0.002' => 263.578,  '0.001' => 268.695],
        202  => ['0.995' => 153.984,   '0.975' => 164.532,  '0.200' => 218.693,  '0.100' => 228.149,  '0.050' => 236.159,  '0.025' => 243.254,  '0.020' => 245.392,  '0.010' => 251.677,  '0.005' => 257.521,  '0.002' => 264.721,  '0.001' => 269.849],
        203  => ['0.995' => 154.856,   '0.975' => 165.435,  '0.200' => 219.735,  '0.100' => 229.213,  '0.050' => 237.240,  '0.025' => 244.351,  '0.020' => 246.494,  '0.010' => 252.793,  '0.005' => 258.649,  '0.002' => 265.864,  '0.001' => 271.002],
        204  => ['0.995' => 155.728,   '0.975' => 166.338,  '0.200' => 220.777,  '0.100' => 230.276,  '0.050' => 238.322,  '0.025' => 245.448,  '0.020' => 247.596,  '0.010' => 253.908,  '0.005' => 259.777,  '0.002' => 267.007,  '0.001' => 272.155],
        205  => ['0.995' => 156.601,   '0.975' => 167.241,  '0.200' => 221.818,  '0.100' => 231.340,  '0.050' => 239.403,  '0.025' => 246.545,  '0.020' => 248.698,  '0.010' => 255.023,  '0.005' => 260.904,  '0.002' => 268.149,  '0.001' => 273.308],
        206  => ['0.995' => 157.474,   '0.975' => 168.144,  '0.200' => 222.860,  '0.100' => 232.403,  '0.050' => 240.485,  '0.025' => 247.642,  '0.020' => 249.799,  '0.010' => 256.138,  '0.005' => 262.031,  '0.002' => 269.291,  '0.001' => 274.460],
        207  => ['0.995' => 158.347,   '0.975' => 169.047,  '0.200' => 223.901,  '0.100' => 233.466,  '0.050' => 241.566,  '0.025' => 248.739,  '0.020' => 250.900,  '0.010' => 257.253,  '0.005' => 263.158,  '0.002' => 270.432,  '0.001' => 275.612],
        208  => ['0.995' => 159.221,   '0.975' => 169.951,  '0.200' => 224.943,  '0.100' => 234.529,  '0.050' => 242.647,  '0.025' => 249.835,  '0.020' => 252.001,  '0.010' => 258.367,  '0.005' => 264.285,  '0.002' => 271.574,  '0.001' => 276.764],
        209  => ['0.995' => 160.095,   '0.975' => 170.855,  '0.200' => 225.984,  '0.100' => 235.592,  '0.050' => 243.727,  '0.025' => 250.931,  '0.020' => 253.102,  '0.010' => 259.481,  '0.005' => 265.411,  '0.002' => 272.715,  '0.001' => 277.915],
        210  => ['0.995' => 160.969,   '0.975' => 171.759,  '0.200' => 227.025,  '0.100' => 236.655,  '0.050' => 244.808,  '0.025' => 252.027,  '0.020' => 254.202,  '0.010' => 260.595,  '0.005' => 266.537,  '0.002' => 273.855,  '0.001' => 279.066],
        211  => ['0.995' => 161.843,   '0.975' => 172.664,  '0.200' => 228.066,  '0.100' => 237.717,  '0.050' => 245.888,  '0.025' => 253.122,  '0.020' => 255.302,  '0.010' => 261.708,  '0.005' => 267.662,  '0.002' => 274.995,  '0.001' => 280.217],
        212  => ['0.995' => 162.718,   '0.975' => 173.568,  '0.200' => 229.107,  '0.100' => 238.780,  '0.050' => 246.968,  '0.025' => 254.218,  '0.020' => 256.402,  '0.010' => 262.821,  '0.005' => 268.788,  '0.002' => 276.135,  '0.001' => 281.367],
        213  => ['0.995' => 163.593,   '0.975' => 174.473,  '0.200' => 230.148,  '0.100' => 239.842,  '0.050' => 248.048,  '0.025' => 255.313,  '0.020' => 257.502,  '0.010' => 263.934,  '0.005' => 269.912,  '0.002' => 277.275,  '0.001' => 282.517],
        214  => ['0.995' => 164.469,   '0.975' => 175.378,  '0.200' => 231.189,  '0.100' => 240.904,  '0.050' => 249.128,  '0.025' => 256.408,  '0.020' => 258.601,  '0.010' => 265.047,  '0.005' => 271.037,  '0.002' => 278.414,  '0.001' => 283.666],
        215  => ['0.995' => 165.344,   '0.975' => 176.283,  '0.200' => 232.230,  '0.100' => 241.966,  '0.050' => 250.207,  '0.025' => 257.503,  '0.020' => 259.701,  '0.010' => 266.159,  '0.005' => 272.162,  '0.002' => 279.553,  '0.001' => 284.815],
        216  => ['0.995' => 166.220,   '0.975' => 177.189,  '0.200' => 233.270,  '0.100' => 243.028,  '0.050' => 251.286,  '0.025' => 258.597,  '0.020' => 260.800,  '0.010' => 267.271,  '0.005' => 273.286,  '0.002' => 280.692,  '0.001' => 285.964],
        217  => ['0.995' => 167.096,   '0.975' => 178.095,  '0.200' => 234.311,  '0.100' => 244.090,  '0.050' => 252.365,  '0.025' => 259.691,  '0.020' => 261.898,  '0.010' => 268.383,  '0.005' => 274.409,  '0.002' => 281.830,  '0.001' => 287.112],
        218  => ['0.995' => 167.973,   '0.975' => 179.001,  '0.200' => 235.351,  '0.100' => 245.151,  '0.050' => 253.444,  '0.025' => 260.785,  '0.020' => 262.997,  '0.010' => 269.495,  '0.005' => 275.533,  '0.002' => 282.968,  '0.001' => 288.261],
        219  => ['0.995' => 168.850,   '0.975' => 179.907,  '0.200' => 236.391,  '0.100' => 246.213,  '0.050' => 254.523,  '0.025' => 261.879,  '0.020' => 264.095,  '0.010' => 270.606,  '0.005' => 276.656,  '0.002' => 284.106,  '0.001' => 289.408],
        220  => ['0.995' => 169.727,   '0.975' => 180.813,  '0.200' => 237.432,  '0.100' => 247.274,  '0.050' => 255.602,  '0.025' => 262.973,  '0.020' => 265.193,  '0.010' => 271.717,  '0.005' => 277.779,  '0.002' => 285.243,  '0.001' => 290.556],
        221  => ['0.995' => 170.604,   '0.975' => 181.720,  '0.200' => 238.472,  '0.100' => 248.335,  '0.050' => 256.680,  '0.025' => 264.066,  '0.020' => 266.291,  '0.010' => 272.828,  '0.005' => 278.902,  '0.002' => 286.380,  '0.001' => 291.703],
        222  => ['0.995' => 171.482,   '0.975' => 182.627,  '0.200' => 239.512,  '0.100' => 249.396,  '0.050' => 257.758,  '0.025' => 265.159,  '0.020' => 267.389,  '0.010' => 273.939,  '0.005' => 280.024,  '0.002' => 287.517,  '0.001' => 292.850],
        223  => ['0.995' => 172.360,   '0.975' => 183.534,  '0.200' => 240.552,  '0.100' => 250.457,  '0.050' => 258.837,  '0.025' => 266.252,  '0.020' => 268.486,  '0.010' => 275.049,  '0.005' => 281.146,  '0.002' => 288.653,  '0.001' => 293.996],
        224  => ['0.995' => 173.238,   '0.975' => 184.441,  '0.200' => 241.592,  '0.100' => 251.517,  '0.050' => 259.914,  '0.025' => 267.345,  '0.020' => 269.584,  '0.010' => 276.159,  '0.005' => 282.268,  '0.002' => 289.789,  '0.001' => 295.142],
        225  => ['0.995' => 174.116,   '0.975' => 185.348,  '0.200' => 242.631,  '0.100' => 252.578,  '0.050' => 260.992,  '0.025' => 268.438,  '0.020' => 270.681,  '0.010' => 277.269,  '0.005' => 283.390,  '0.002' => 290.925,  '0.001' => 296.288],
        226  => ['0.995' => 174.995,   '0.975' => 186.256,  '0.200' => 243.671,  '0.100' => 253.638,  '0.050' => 262.070,  '0.025' => 269.530,  '0.020' => 271.777,  '0.010' => 278.379,  '0.005' => 284.511,  '0.002' => 292.061,  '0.001' => 297.433],
        227  => ['0.995' => 175.874,   '0.975' => 187.164,  '0.200' => 244.711,  '0.100' => 254.699,  '0.050' => 263.147,  '0.025' => 270.622,  '0.020' => 272.874,  '0.010' => 279.488,  '0.005' => 285.632,  '0.002' => 293.196,  '0.001' => 298.579],
        228  => ['0.995' => 176.753,   '0.975' => 188.072,  '0.200' => 245.750,  '0.100' => 255.759,  '0.050' => 264.224,  '0.025' => 271.714,  '0.020' => 273.970,  '0.010' => 280.597,  '0.005' => 286.753,  '0.002' => 294.331,  '0.001' => 299.723],
        229  => ['0.995' => 177.633,   '0.975' => 188.980,  '0.200' => 246.790,  '0.100' => 256.819,  '0.050' => 265.301,  '0.025' => 272.806,  '0.020' => 275.066,  '0.010' => 281.706,  '0.005' => 287.874,  '0.002' => 295.465,  '0.001' => 300.868],
        230  => ['0.995' => 178.512,   '0.975' => 189.889,  '0.200' => 247.829,  '0.100' => 257.879,  '0.050' => 266.378,  '0.025' => 273.898,  '0.020' => 276.162,  '0.010' => 282.814,  '0.005' => 288.994,  '0.002' => 296.600,  '0.001' => 302.012],
        231  => ['0.995' => 179.392,   '0.975' => 190.797,  '0.200' => 248.868,  '0.100' => 258.939,  '0.050' => 267.455,  '0.025' => 274.989,  '0.020' => 277.258,  '0.010' => 283.923,  '0.005' => 290.114,  '0.002' => 297.734,  '0.001' => 303.156],
        232  => ['0.995' => 180.273,   '0.975' => 191.706,  '0.200' => 249.908,  '0.100' => 259.998,  '0.050' => 268.531,  '0.025' => 276.080,  '0.020' => 278.354,  '0.010' => 285.031,  '0.005' => 291.234,  '0.002' => 298.867,  '0.001' => 304.299],
        233  => ['0.995' => 181.153,   '0.975' => 192.615,  '0.200' => 250.947,  '0.100' => 261.058,  '0.050' => 269.608,  '0.025' => 277.171,  '0.020' => 279.449,  '0.010' => 286.139,  '0.005' => 292.353,  '0.002' => 300.001,  '0.001' => 305.443],
        234  => ['0.995' => 182.034,   '0.975' => 193.524,  '0.200' => 251.986,  '0.100' => 262.117,  '0.050' => 270.684,  '0.025' => 278.262,  '0.020' => 280.544,  '0.010' => 287.247,  '0.005' => 293.472,  '0.002' => 301.134,  '0.001' => 306.586],
        235  => ['0.995' => 182.915,   '0.975' => 194.434,  '0.200' => 253.025,  '0.100' => 263.176,  '0.050' => 271.760,  '0.025' => 279.352,  '0.020' => 281.639,  '0.010' => 288.354,  '0.005' => 294.591,  '0.002' => 302.267,  '0.001' => 307.728],
        236  => ['0.995' => 183.796,   '0.975' => 195.343,  '0.200' => 254.063,  '0.100' => 264.235,  '0.050' => 272.836,  '0.025' => 280.443,  '0.020' => 282.734,  '0.010' => 289.461,  '0.005' => 295.710,  '0.002' => 303.400,  '0.001' => 308.871],
        237  => ['0.995' => 184.678,   '0.975' => 196.253,  '0.200' => 255.102,  '0.100' => 265.294,  '0.050' => 273.911,  '0.025' => 281.533,  '0.020' => 283.828,  '0.010' => 290.568,  '0.005' => 296.828,  '0.002' => 304.532,  '0.001' => 310.013],
        238  => ['0.995' => 185.560,   '0.975' => 197.163,  '0.200' => 256.141,  '0.100' => 266.353,  '0.050' => 274.987,  '0.025' => 282.623,  '0.020' => 284.922,  '0.010' => 291.675,  '0.005' => 297.947,  '0.002' => 305.664,  '0.001' => 311.154],
        239  => ['0.995' => 186.442,   '0.975' => 198.073,  '0.200' => 257.179,  '0.100' => 267.412,  '0.050' => 276.062,  '0.025' => 283.713,  '0.020' => 286.016,  '0.010' => 292.782,  '0.005' => 299.065,  '0.002' => 306.796,  '0.001' => 312.296],
        240  => ['0.995' => 187.324,   '0.975' => 198.984,  '0.200' => 258.218,  '0.100' => 268.471,  '0.050' => 277.138,  '0.025' => 284.802,  '0.020' => 287.110,  '0.010' => 293.888,  '0.005' => 300.182,  '0.002' => 307.927,  '0.001' => 313.437],
        241  => ['0.995' => 188.207,   '0.975' => 199.894,  '0.200' => 259.256,  '0.100' => 269.529,  '0.050' => 278.213,  '0.025' => 285.892,  '0.020' => 288.204,  '0.010' => 294.994,  '0.005' => 301.300,  '0.002' => 309.058,  '0.001' => 314.578],
        242  => ['0.995' => 189.090,   '0.975' => 200.805,  '0.200' => 260.295,  '0.100' => 270.588,  '0.050' => 279.288,  '0.025' => 286.981,  '0.020' => 289.298,  '0.010' => 296.100,  '0.005' => 302.417,  '0.002' => 310.189,  '0.001' => 315.718],
        243  => ['0.995' => 189.973,   '0.975' => 201.716,  '0.200' => 261.333,  '0.100' => 271.646,  '0.050' => 280.362,  '0.025' => 288.070,  '0.020' => 290.391,  '0.010' => 297.206,  '0.005' => 303.534,  '0.002' => 311.320,  '0.001' => 316.859],
        244  => ['0.995' => 190.856,   '0.975' => 202.627,  '0.200' => 262.371,  '0.100' => 272.704,  '0.050' => 281.437,  '0.025' => 289.159,  '0.020' => 291.484,  '0.010' => 298.311,  '0.005' => 304.651,  '0.002' => 312.450,  '0.001' => 317.999],
        245  => ['0.995' => 191.739,   '0.975' => 203.539,  '0.200' => 263.409,  '0.100' => 273.762,  '0.050' => 282.511,  '0.025' => 290.248,  '0.020' => 292.577,  '0.010' => 299.417,  '0.005' => 305.767,  '0.002' => 313.580,  '0.001' => 319.138],
        246  => ['0.995' => 192.623,   '0.975' => 204.450,  '0.200' => 264.447,  '0.100' => 274.820,  '0.050' => 283.586,  '0.025' => 291.336,  '0.020' => 293.670,  '0.010' => 300.522,  '0.005' => 306.883,  '0.002' => 314.710,  '0.001' => 320.278],
        247  => ['0.995' => 193.507,   '0.975' => 205.362,  '0.200' => 265.485,  '0.100' => 275.878,  '0.050' => 284.660,  '0.025' => 292.425,  '0.020' => 294.762,  '0.010' => 301.626,  '0.005' => 307.999,  '0.002' => 315.840,  '0.001' => 321.417],
        248  => ['0.995' => 194.391,   '0.975' => 206.274,  '0.200' => 266.523,  '0.100' => 276.935,  '0.050' => 285.734,  '0.025' => 293.513,  '0.020' => 295.855,  '0.010' => 302.731,  '0.005' => 309.115,  '0.002' => 316.969,  '0.001' => 322.556],
        249  => ['0.995' => 195.276,   '0.975' => 207.186,  '0.200' => 267.561,  '0.100' => 277.993,  '0.050' => 286.808,  '0.025' => 294.601,  '0.020' => 296.947,  '0.010' => 303.835,  '0.005' => 310.231,  '0.002' => 318.098,  '0.001' => 323.694],
        250  => ['0.995' => 196.161,   '0.975' => 208.098,  '0.200' => 268.599,  '0.100' => 279.050,  '0.050' => 287.882,  '0.025' => 295.689,  '0.020' => 298.039,  '0.010' => 304.940,  '0.005' => 311.346,  '0.002' => 319.227,  '0.001' => 324.832],
        300  => ['0.995' => 240.663,   '0.975' => 253.912,  '0.200' => 320.397,  '0.100' => 331.789,  '0.050' => 341.395,  '0.025' => 349.874,  '0.020' => 352.425,  '0.010' => 359.906,  '0.005' => 366.844,  '0.002' => 375.369,  '0.001' => 381.425],
        350  => ['0.995' => 285.608,   '0.975' => 300.064,  '0.200' => 372.051,  '0.100' => 384.306,  '0.050' => 394.626,  '0.025' => 403.723,  '0.020' => 406.457,  '0.010' => 414.474,  '0.005' => 421.900,  '0.002' => 431.017,  '0.001' => 437.488],
        400  => ['0.995' => 330.903,   '0.975' => 346.482,  '0.200' => 423.590,  '0.100' => 436.649,  '0.050' => 447.632,  '0.025' => 457.305,  '0.020' => 460.211,  '0.010' => 468.724,  '0.005' => 476.606,  '0.002' => 486.274,  '0.001' => 493.132],
        450  => ['0.995' => 376.483,   '0.975' => 393.118,  '0.200' => 475.035,  '0.100' => 488.849,  '0.050' => 500.456,  '0.025' => 510.670,  '0.020' => 513.736,  '0.010' => 522.717,  '0.005' => 531.026,  '0.002' => 541.212,  '0.001' => 548.432],
        500  => ['0.995' => 422.303,   '0.975' => 439.936,  '0.200' => 526.401,  '0.100' => 540.930,  '0.050' => 553.127,  '0.025' => 563.852,  '0.020' => 567.070,  '0.010' => 576.493,  '0.005' => 585.207,  '0.002' => 595.882,  '0.001' => 603.446],
        550  => ['0.995' => 468.328,   '0.975' => 486.910,  '0.200' => 577.701,  '0.100' => 592.909,  '0.050' => 605.667,  '0.025' => 616.878,  '0.020' => 620.241,  '0.010' => 630.084,  '0.005' => 639.183,  '0.002' => 650.324,  '0.001' => 658.215],
        600  => ['0.995' => 514.529,   '0.975' => 534.019,  '0.200' => 628.943,  '0.100' => 644.800,  '0.050' => 658.094,  '0.025' => 669.769,  '0.020' => 673.270,  '0.010' => 683.516,  '0.005' => 692.982,  '0.002' => 704.568,  '0.001' => 712.771],
        650  => ['0.995' => 560.885,   '0.975' => 581.245,  '0.200' => 680.134,  '0.100' => 696.614,  '0.050' => 710.421,  '0.025' => 722.542,  '0.020' => 726.176,  '0.010' => 736.807,  '0.005' => 746.625,  '0.002' => 758.639,  '0.001' => 767.141],
        700  => ['0.995' => 607.380,   '0.975' => 628.577,  '0.200' => 731.280,  '0.100' => 748.359,  '0.050' => 762.661,  '0.025' => 775.211,  '0.020' => 778.972,  '0.010' => 789.974,  '0.005' => 800.131,  '0.002' => 812.556,  '0.001' => 821.347],
        750  => ['0.995' => 653.997,   '0.975' => 676.003,  '0.200' => 782.386,  '0.100' => 800.043,  '0.050' => 814.822,  '0.025' => 827.785,  '0.020' => 831.670,  '0.010' => 843.029,  '0.005' => 853.514,  '0.002' => 866.336,  '0.001' => 875.404],
        800  => ['0.995' => 700.725,   '0.975' => 723.513,  '0.200' => 833.456,  '0.100' => 851.671,  '0.050' => 866.911,  '0.025' => 880.275,  '0.020' => 884.279,  '0.010' => 895.984,  '0.005' => 906.786,  '0.002' => 919.991,  '0.001' => 929.329],
        850  => ['0.995' => 747.554,   '0.975' => 771.099,  '0.200' => 884.492,  '0.100' => 903.249,  '0.050' => 918.937,  '0.025' => 932.689,  '0.020' => 936.808,  '0.010' => 948.848,  '0.005' => 959.957,  '0.002' => 973.534,  '0.001' => 983.133],
        900  => ['0.995' => 794.475,   '0.975' => 818.756,  '0.200' => 935.499,  '0.100' => 954.782,  '0.050' => 970.904,  '0.025' => 985.032,  '0.020' => 989.263,  '0.010' => 1001.630, '0.005' => 1013.036, '0.002' => 1026.974, '0.001' => 1036.826],
        950  => ['0.995' => 841.480,   '0.975' => 866.477,  '0.200' => 986.478,  '0.100' => 1006.272, '0.050' => 1022.816, '0.025' => 1037.311, '0.020' => 1041.651, '0.010' => 1054.334, '0.005' => 1066.031, '0.002' => 1080.320, '0.001' => 1090.418],
        1000 => ['0.995' => 888.564,   '0.975' => 914.257,  '0.200' => 1037.431, '0.100' => 1057.724, '0.050' => 1074.679, '0.025' => 1089.531, '0.020' => 1093.977, '0.010' => 1106.969, '0.005' => 1118.948, '0.002' => 1133.579, '0.001' => 1143.917],
    ];

    /**
     * Get χ² score
     *
     * @param int   $df Degrees of freedom
     * @param float $p P value
     *
     * @return float χ² score
     *
     * @throws Exception\BadDataException
     */
    public static function getChiSquareValue(int $df, float $p): float
    {
        if (isset(self::CHI_SQUARED_SCORES[$df][\sprintf('%1.3f', $p)])) {
            return self::CHI_SQUARED_SCORES[$df][\sprintf('%1.3f', $p)];
        }

        throw new Exception\BadDataException("No chi-squared value for degrees of freedom $df and p value " . \sprintf('%1.3f', $p));
    }
}
