<?php

namespace Modules\AAdvantageMiles\Http\Controllers;

use App\Contracts\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Modules\AAdvantageMiles\Models\MilesHistory;
use Modules\AAdvantageMiles\Models\MilesSettings;
use App\Models\User;

class AdminController extends Controller
{
    public function index()
    {
        $settings = MilesSettings::firstOrNew(['id' => 1]);
        return view('AAMiles::admin.settings', compact('settings'));
    }

    public function save(Request $request)
    {
        $settings = MilesSettings::firstOrNew(['id' => 1]);
        $data = $request->validate([
            // Checkboxes come through as "on"; don't use the boolean rule
            'enabled' => 'nullable',
            'daily_visit_enabled' => 'nullable',
            'daily_visit_miles' => 'nullable|integer',
            'miles_per_min_enabled' => 'nullable',
            'miles_per_min' => 'nullable|integer',
            'smooth_enabled' => 'nullable',
            'smooth_limit' => 'nullable|integer',
            'smooth_miles' => 'nullable|integer',
            'bad_enabled' => 'nullable',
            'bad_limit' => 'nullable|integer',
            'bad_miles' => 'nullable|integer',
            'score_bonus_enabled' => 'nullable',
            'score_max' => 'nullable|integer',
            'airport_miles' => 'nullable',
            'shop_airline_id' => 'nullable|integer|min:1',
            'enforce_rank_bids' => 'nullable',
        ]);

        // Normalize checkboxes
        foreach (['enabled','daily_visit_enabled','miles_per_min_enabled','smooth_enabled','bad_enabled','score_bonus_enabled','enforce_rank_bids'] as $cb) {
            $data[$cb] = (int)($request->boolean($cb));
        }
        // Default airline for shop flight search (for Flight Pass tooling)
        if (!isset($data['shop_airline_id']) || (int)$data['shop_airline_id'] < 1) {
            $data['shop_airline_id'] = (int)(\App\Models\Airline::min('id') ?: 1);
        }

        // Accept airport_miles as JSON text or array, and normalize rows
        $airport_miles = $request->input('airport_miles');
        $rows = [];
        if (is_string($airport_miles)) {
            $decoded = json_decode($airport_miles, true);
            $rows = is_array($decoded) ? $decoded : [];
        } elseif (is_array($airport_miles)) {
            $rows = $airport_miles;
        }
        $normalized = [];
        foreach ($rows as $r) {
            $icao = strtoupper(trim((string)($r['icao'] ?? '')));
            $miles = (int)($r['miles'] ?? 0);
            $when = strtolower(trim((string)($r['when'] ?? 'both')));
            if (!in_array($when, ['departure','arrival','both'], true)) { $when = 'both'; }
            if ($icao !== '' && $miles > 0) {
                $normalized[] = ['icao' => $icao, 'miles' => $miles, 'when' => $when];
            }
        }
        $data['airport_miles'] = $normalized;

        $settings->fill($data);
        $settings->save();

        return redirect()->back()->with('success', 'Settings saved');
    }

    // Single-page Manage Pilot Miles: search + adjust
    public function manage(Request $request)
    {
        $q = trim((string)$request->get('q'));
        $user = null;
        $balance = null;
        $recent = collect();

        if ($q !== '') {
            $user = User::query()
                ->where('name', 'like', "%$q%")
                ->orWhere('email', 'like', "%$q%")
                ->orWhere('callsign', 'like', "%$q%")
                ->when(preg_match('/\d+/', $q), function ($qB) use ($q) {
                    $digits = preg_replace('/\D+/', '', $q);
                    if ($digits !== '') {
                        $qB->orWhere('pilot_id', (int)$digits);
                    }
                })
                ->first();

            if ($user) {
                $balance = MilesHistory::selectRaw('SUM(miles) as total, SUM(CASE WHEN miles>=0 THEN miles ELSE 0 END) as earned, SUM(CASE WHEN miles<0 THEN miles ELSE 0 END) as spent')
                    ->where('user_id', $user->id)
                    ->first();

                $recent = MilesHistory::where('user_id', $user->id)
                    ->orderByDesc('created_at')
                    ->limit(10)
                    ->get();
            }
        }

        return view('AAMiles::admin.manage', compact('q', 'user', 'balance', 'recent'));
    }

    public function adjust(Request $request)
    {
        $data = $request->validate([
            'user_id' => 'required|integer|exists:users,id',
            'amount' => 'required|integer|not_in:0',
            'note' => 'required|string|max:200',
        ]);

        $adminId = Auth::id() ?? 0;

        MilesHistory::create([
            'user_id' => (int)$data['user_id'],
            'issued_by' => $adminId,
            'source' => 'admin-adjust',
            'event_id' => (string)$data['user_id'],
            'note' => $data['note'],
            'miles' => (int)$data['amount'],
            'meta' => [],
            'created_at' => now(),
        ]);

        return redirect()->route('AAMiles.admin.manage', ['q' => $request->get('return_q')])
            ->with('success', 'Miles updated successfully');
    }
}

