<?php

namespace Modules\AAdvantageMiles\Services;

use Modules\AAdvantageMiles\Models\ShopOrder;
use Modules\AAdvantageMiles\Models\ShopProduct;
use Modules\AAdvantageMiles\Models\FlightPass;
use App\Models\User;

class FlightPassService
{
    /**
     * Find an active, unused Flight Pass record for this user/flight.
     */
    public function findActivePassForFlight(int $userId, $flightId): ?FlightPass
    {
        $now = now();
        return FlightPass::where('user_id', $userId)
            ->where('flight_id', (string)$flightId)
            ->where(function ($q) use ($now) {
                $q->whereNull('expires_at')->orWhere('expires_at', '>=', $now);
            })
            ->whereNull('used_at')
            ->where(function ($q) {
                $q->whereNull('status')->orWhere('status', 'active');
            })
            ->orderByDesc('created_at')
            ->first();
    }

    /**
     * For compatibility with existing callers, return the associated ShopOrder
     * for an active, unused pass if any.
     */
    public function getUnusedPassOrderForFlight(int $userId, $flightId): ?ShopOrder
    {
        $pass = $this->findActivePassForFlight($userId, $flightId);
        if (!$pass) { return null; }
        if (empty($pass->order_id)) { return null; }
        return ShopOrder::find($pass->order_id);
    }

    public function allowsForFlight(User $user, $flightId): bool
    {
        if (empty($flightId)) { return false; }
        return $this->findActivePassForFlight($user->id, $flightId) !== null;
    }

    public function markUsed(User $user, $flightId, ?string $pirepId = null): void
    {
        if (empty($flightId)) { return; }
        $pass = $this->findActivePassForFlight($user->id, (string)$flightId);
        if ($pass) {
            $pass->used_at = now();
            $pass->status = 'used';
            $pass->save();

            // Best-effort: also mark the linked shop order meta for UI backward compatibility
            if (!empty($pass->order_id)) {
                $order = ShopOrder::find($pass->order_id);
                if ($order) {
                    $meta = is_array($order->meta ?? null) ? $order->meta : [];
                    $meta['used'] = true;
                    if (!empty($pirepId)) { $meta['pirep_id'] = $pirepId; }
                    $meta['used_at'] = now()->toDateTimeString();
                    $order->meta = $meta;
                    $order->save();
                }
            }
        }
    }
}

