<?php

use App\Contracts\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class CreateAadvantageMilesTables extends Migration
{
    public function up()
    {
        // Settings
        if (!Schema::hasTable('aadv_miles_settings')) {
            Schema::create('aadv_miles_settings', function (Blueprint $table) {
                $table->increments('id');
                $table->boolean('enabled')->default(false);
                $table->boolean('daily_visit_enabled')->default(false);
                $table->integer('daily_visit_miles')->default(0);
                $table->boolean('miles_per_min_enabled')->default(false);
                $table->integer('miles_per_min')->default(0);
                $table->boolean('smooth_enabled')->default(false);
                $table->integer('smooth_limit')->default(-200); // fpm threshold (higher than this and <0 means smooth)
                $table->integer('smooth_miles')->default(0);
                $table->boolean('bad_enabled')->default(false);
                $table->integer('bad_limit')->default(-500); // fpm threshold (lower than this is bad)
                $table->integer('bad_miles')->default(0); // subtract if set
                $table->boolean('score_bonus_enabled')->default(false);
                $table->integer('score_max')->default(0);
                $table->json('airport_miles')->nullable(); // [{icao,miles,when}]
            });

            // Seed defaults
            DB::table('aadv_miles_settings')->insert([
                'id' => 1,
                'enabled' => true,
                'daily_visit_enabled' => true,
                'daily_visit_miles' => 20,
                'miles_per_min_enabled' => true,
                'miles_per_min' => 2,
                'smooth_enabled' => true,
                'smooth_limit' => -100,
                'smooth_miles' => 500,
                'bad_enabled' => true,
                'bad_limit' => -500,
                'bad_miles' => 200,
                'score_bonus_enabled' => true,
                'score_max' => 1000,
                'airport_miles' => json_encode([]),
            ]);
        }

        // Ledger/history
        if (!Schema::hasTable('aadv_miles_history')) {
            Schema::create('aadv_miles_history', function (Blueprint $table) {
                $table->increments('id');
                $table->unsignedInteger('user_id')->index();
                $table->unsignedInteger('issued_by')->nullable()->index();
                $table->string('source', 64)->index();
                $table->string('event_id', 64)->nullable();
                $table->string('note', 200)->nullable();
                $table->integer('miles');
                $table->json('meta')->nullable();
                $table->string('unique_key', 150)->nullable()->unique();
                $table->dateTime('created_at')->default(DB::raw('CURRENT_TIMESTAMP'))->index();
            });
        }

        // Shop categories
        if (!Schema::hasTable('aadv_shop_categories')) {
            Schema::create('aadv_shop_categories', function (Blueprint $table) {
                $table->increments('id');
                $table->string('name', 120);
                $table->integer('sort')->default(0);
                $table->boolean('is_active')->default(true);
            });
        }

        // Shop products
        if (!Schema::hasTable('aadv_shop_products')) {
            Schema::create('aadv_shop_products', function (Blueprint $table) {
                $table->increments('id');
                $table->unsignedInteger('category_id');
                $table->string('name', 160);
                $table->text('description')->nullable();
                $table->string('image_url', 255)->nullable();
                $table->integer('cost_miles');
                $table->boolean('is_active')->default(true);
                $table->integer('sort')->default(0);
                $table->integer('stock_limit')->nullable();
                $table->integer('per_user_limit')->nullable();
                $table->json('meta')->nullable();
                $table->index('category_id');
                $table->foreign('category_id')->references('id')->on('aadv_shop_categories')->onDelete('cascade');
            });
        }

        // Shop orders
        if (!Schema::hasTable('aadv_shop_orders')) {
            Schema::create('aadv_shop_orders', function (Blueprint $table) {
                $table->increments('id');
                $table->unsignedInteger('user_id');
                $table->unsignedInteger('product_id');
                $table->integer('cost_miles');
                $table->json('meta')->nullable();
                $table->dateTime('created_at')->default(DB::raw('CURRENT_TIMESTAMP'))->index();
                $table->index('user_id');
                $table->index('product_id');
                $table->foreign('product_id')->references('id')->on('aadv_shop_products')->onDelete('cascade');
                // User FK can be added if desired; leaving as index for broader compatibility
            });
        }

        // Flight Passes (for direct bid)
        if (!Schema::hasTable('aadv_flight_passes')) {
            Schema::create('aadv_flight_passes', function (Blueprint $table) {
                $table->increments('id');
                $table->unsignedInteger('user_id')->index();
                $table->string('flight_id', 64)->index();
                $table->unsignedInteger('aircraft_id')->nullable()->index();
                $table->unsignedInteger('order_id')->nullable()->index();
                $table->string('status', 32)->nullable()->index();
                $table->dateTime('expires_at')->nullable()->index();
                $table->dateTime('used_at')->nullable()->index();
                $table->dateTime('created_at')->default(DB::raw('CURRENT_TIMESTAMP'))->index();
                $table->foreign('order_id')->references('id')->on('aadv_shop_orders')->onDelete('set null');
            });
        }
    }

    public function down()
    {
        Schema::dropIfExists('aadv_flight_passes');
        Schema::dropIfExists('aadv_shop_orders');
        Schema::dropIfExists('aadv_shop_products');
        Schema::dropIfExists('aadv_shop_categories');
        Schema::dropIfExists('aadv_miles_history');
        Schema::dropIfExists('aadv_miles_settings');
    }
}

