<?php

namespace Modules\AAdvantageMiles\Http\Controllers;

use App\Contracts\Controller;
use App\Models\Flight;
use Illuminate\Http\Request;
use Modules\AAdvantageMiles\Models\ShopCategory;
use Modules\AAdvantageMiles\Models\ShopProduct;
use Modules\AAdvantageMiles\Models\ShopOrder;

class AdminShopController extends Controller
{
    public function index(Request $request)
    {
        $categories = ShopCategory::orderBy('sort')->get();
        $products = ShopProduct::orderBy('sort')->get();
        $edit = null;
        $editCat = null;
        if ($request->has('edit')) {
            $edit = ShopProduct::find($request->query('edit'));
        }
        $editCatParam = $request->query('edit_cat', null);
        if ($editCatParam !== null && $editCatParam !== '') {
            $editCat = ShopCategory::find($editCatParam);
        }
        // Seed airline for admin flight search (lock to one airline)
        $seed_airline_id = (int)($request->get('airline_id') ?? optional(\Modules\AAdvantageMiles\Models\MilesSettings::first())->shop_airline_id ?? (\App\Models\Airline::min('id') ?: 1));
        $sel_flight_id = (string)$request->query('flight_id', '');

        // Recent purchases ledger (joined with users and products)
        $orders = ShopOrder::query()
            ->select(['aadv_shop_orders.*', 'users.name as user_name', 'users.id as user_id', 'aadv_shop_products.name as product_name'])
            ->join('users', 'users.id', '=', 'aadv_shop_orders.user_id')
            ->join('aadv_shop_products', 'aadv_shop_products.id', '=', 'aadv_shop_orders.product_id')
            ->orderByDesc('aadv_shop_orders.created_at')
            ->paginate(25);

        return view('AAMiles::admin.shop', compact('categories', 'products', 'edit', 'editCat', 'seed_airline_id', 'sel_flight_id', 'orders'));
    }

    public function saveCategory(Request $request)
    {
        $data = $request->validate([
            'id' => 'nullable|integer',
            'name' => 'required|string|max:120',
            'sort' => 'nullable|integer',
        ]);

        // Update existing category
        if (!empty($data['id'])) {
            $cat = ShopCategory::find($data['id']);
            if ($cat) {
                $cat->update([
                    'name' => $data['name'],
                    'sort' => $data['sort'] ?? 0,
                ]);
                return redirect()->route('AAMiles.admin.shop')->with('success', 'Category updated');
            }
        }

        // Create new category
        $create = [
            'name' => $data['name'],
            'sort' => $data['sort'] ?? 0,
            'is_active' => 1,
        ];
        ShopCategory::create($create);
        return back()->with('success', 'Category added');
    }

    public function saveProduct(Request $request)
    {
        $data = $request->validate([
            'id' => 'nullable|integer',
            'name' => 'required|string|max:160',
            'category_id' => 'required|integer',
            'description' => 'nullable|string',
            'image_url' => 'nullable|string',
            'cost_miles' => 'required|integer|min:1',
            'sort' => 'nullable|integer',
            'stock_limit' => 'nullable|integer',
            'per_user_limit' => 'nullable|integer',
            'is_active' => 'nullable',
            // Flight Pass fields (optional)
            'fp_enabled' => 'nullable',
            // Flight IDs in phpVMS can be strings
            'fp_flight_id' => 'nullable|string',
            // Alternate manual input for flight selection
            'fp_flight_code' => 'nullable|string',
            // Aircraft ID to bind to the bid (optional but recommended for SB)
            'fp_aircraft_id' => 'nullable|integer',
        ]);
        $data['is_active'] = (int)$request->boolean('is_active');

        // Build/resolve meta for Flight Pass if enabled (no JS picker required)
        $meta = [];
        if ($request->boolean('fp_enabled')) {
            $airlineLock = (int)(optional(\Modules\AAdvantageMiles\Models\MilesSettings::first())->shop_airline_id ?? (\App\Models\Airline::min('id') ?: 0));
            $inId = trim((string)$request->input('fp_flight_id'));
            $inCode = trim((string)$request->input('fp_flight_code'));
            $resolvedId = null;

            // Prefer the newly typed code/number when present
            if ($inCode !== '') {
                // Extract numeric portion of flight number from inputs like "AA123", "AA123/C/L.1", etc
                if (preg_match('/([0-9]+)/', $inCode, $m)) {
                    $num = $m[1];
                    $resolvedId = Flight::where('flight_number', $num)
                        ->when($airlineLock > 0, function($q) use ($airlineLock){ $q->where('airline_id', $airlineLock); })
                        ->orderBy('id')
                        ->value('id');
                }
                // If not resolved by number, allow direct ID pasted into this field
                if (!$resolvedId) {
                    $resolvedId = Flight::when($airlineLock > 0, function($q) use ($airlineLock){ $q->where('airline_id', $airlineLock); })->where('id', $inCode)->value('id');
                }
            }
            // Fallback: keep existing hidden ID if no code was provided
            if (!$resolvedId && $inId !== '') {
                $resolvedId = Flight::where('airline_id', $airlineLock)->where('id', $inId)->value('id');
            }

            if ($resolvedId) {
                $meta['type'] = 'flight_pass';
                $meta['flight_id'] = $resolvedId;
                // Optional aircraft id to pre-bind for SimBrief
                $aircraftId = (int)$request->input('fp_aircraft_id');
                if ($aircraftId > 0) { $meta['aircraft_id'] = $aircraftId; }
            } else {
                return back()->withErrors(['fp_flight_code' => 'Flight not found. Enter a flight number (e.g., 123) or paste the Flight ID.'])->withInput();
            }
        }

        if (!empty($data['id'])) {
            $product = ShopProduct::find($data['id']);
            if ($product) {
                unset($data['id']);
                // Merge/replace meta
                if (!empty($meta)) { $data['meta'] = $meta; }
                $product->update($data);
                // After updating, reset the form state by redirecting without the edit param
                return redirect()->route('AAMiles.admin.shop')->with('success', 'Product updated');
            }
        }





        unset($data['id']);
        if (!empty($meta)) { $data['meta'] = $meta; }
        ShopProduct::create($data);
        return back()->with('success', 'Product saved');
    }

    // JSON endpoint for Select2 to search ACC flights
    public function flightSearch(Request $request)
    {
        $q = trim((string)$request->get('q', ''));
        $page = max(1, (int)$request->get('page', 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $airline_id = (int)($request->get('airline_id') ?? optional(\Modules\AAdvantageMiles\Models\MilesSettings::first())->shop_airline_id ?? (\App\Models\Airline::min('id') ?: 1));

        $builder = Flight::with('airline')
            ->where('active', 1)
            ->where('airline_id', $airline_id);

        if ($q !== '') {
            $builder->where(function ($sub) use ($q) {
                $sub->where('id', $q) // allow direct lookup by flight ID
                    ->orWhere('flight_number', 'like', "%$q%")
                    ->orWhere('dpt_airport_id', 'like', "%$q%")
                    ->orWhere('arr_airport_id', 'like', "%$q%")
                    ->orWhere('route_code', 'like', "%$q%")
                    ->orWhereHas('airline', function ($a) use ($q) { $a->where('icao', 'like', "%$q%"); });
            });
        }

        $total = (clone $builder)->count();
        $rows = $builder->orderBy('flight_number')->skip($offset)->take($perPage)->get();

        $results = $rows->map(function ($f) {
            $al = optional($f->airline)->icao ?: '';
            $text = trim(sprintf('%s%s %s → %s', $al, $f->flight_number, $f->dpt_airport_id, $f->arr_airport_id));
            return ['id' => $f->id, 'text' => $text];
        });

        return response()->json([
            'results' => $results,
            'pagination' => ['more' => ($offset + $perPage) < $total],
        ]);
    }

    // Delete a product (admin)
    public function deleteProduct(Request $request)
    {
        $request->validate(['id' => 'required|integer']);
        $prod = ShopProduct::find($request->input('id'));
        if (!$prod) { return back()->with('error', 'Product not found'); }
        $prod->delete();
        return back()->with('success', 'Product deleted');
    }
}

