<?php

namespace Modules\AAdvantageMiles\Http\Controllers;

use App\Contracts\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Modules\AAdvantageMiles\Models\MilesHistory;
use App\Repositories\FlightRepository;
use App\Services\BidService;
use Illuminate\Http\Request;


class FrontendController extends Controller
{
    public function myMiles()
    {
        $user = Auth::user();

        $totals = MilesHistory::selectRaw('SUM(CASE WHEN miles<0 THEN miles ELSE 0 END) as spent, SUM(CASE WHEN miles>=0 THEN miles ELSE 0 END) as earned, SUM(miles) as total')
            ->where('user_id', $user->id)
            ->first();

        $monthly = MilesHistory::selectRaw('SUM(miles) as total')
            ->where('user_id', $user->id)
            ->whereRaw('YEAR(created_at)=YEAR(CURDATE()) AND MONTH(created_at)=MONTH(CURDATE())')
            ->first();

        $history = MilesHistory::where('user_id', $user->id)
            ->orderByDesc('created_at')
            ->paginate(25);

        return view('AAMiles::frontend.my_miles', compact('totals', 'monthly', 'history'));
    }

    public function leaderboard()
    {
        $all_time = MilesHistory::selectRaw('user_id, SUM(miles) as total')
            ->groupBy('user_id')
            ->orderByDesc('total')
            ->limit(50)
            ->get();

        $monthly = MilesHistory::selectRaw('user_id, SUM(miles) as total')
            ->whereRaw('YEAR(created_at)=YEAR(CURDATE()) AND MONTH(created_at)=MONTH(CURDATE())')
            ->groupBy('user_id')
            ->orderByDesc('total')
            ->limit(50)
            ->get();

        $recent = MilesHistory::orderByDesc('created_at')->limit(100)->get();

        return view('AAMiles::frontend.leaderboard', compact('all_time', 'monthly', 'recent'));
        }


    public function bidPass(\Illuminate\Http\Request $request)
    {
        $user = \Illuminate\Support\Facades\Auth::user();
        $request->validate(['flight_id' => 'required|string|max:36']);
        $flight_id = (string)$request->input('flight_id');

        // Verify the user has an unused pass for this flight
        $fps = new \Modules\AAdvantageMiles\Services\FlightPassService();
        if (!$fps->allowsForFlight($user, $flight_id)) {
            return back()->with('error', 'No valid Flight Pass for this flight');
        }

        try {
            $flight = app(\App\Repositories\FlightRepository::class)->find($flight_id);

            // Try to pre-bind the aircraft from the active Flight Pass (or product meta)
            $aircraft = null;
            try {
                $pass = $fps->findActivePassForFlight($user->id, $flight_id);
                $aid = (int)($pass->aircraft_id ?? 0);
                if ($aid > 0) {
                    $aircraft = \App\Models\Aircraft::find($aid);
                } else {
                    // Fallback: derive from the product meta of the unused pass order
                    $order = $fps->getUnusedPassOrderForFlight($user->id, $flight_id);
                    if ($order) {
                        $product = \Modules\AAdvantageMiles\Models\ShopProduct::find($order->product_id);
                        if ($product) {
                            $pmeta = is_array($product->meta ?? null) ? $product->meta : [];
                            $aid2 = (int)($pmeta['aircraft_id'] ?? 0);
                            if ($aid2 > 0) { $aircraft = \App\Models\Aircraft::find($aid2); }
                        }
                    }
                }
            } catch (\Throwable $e) { /* ignore and fall back to null */ }

            // Optional enforcement of rank/typerating restrictions on Flight Pass bids
            $settings = \Modules\AAdvantageMiles\Models\MilesSettings::first();
            if ($settings && $settings->enforce_rank_bids) {
                // Only allow if user's allowable subfleets include the aircraft's subfleet
                if ($aircraft) {
                    $allowed = app(\App\Services\UserService::class)->aircraftAllowed($user, $aircraft->id);
                    if (!$allowed) {
                        return back()->with('error', 'Your rank/ratings do not allow this aircraft for this bid');
                    }
                }
                // If flight has subfleets, ensure aircraft subfleet is in that list
                $flight_subfleet_ids = $flight->subfleets()->pluck('subfleets.id')->toArray();
                if (!empty($flight_subfleet_ids) && $aircraft) {
                    if (!in_array($aircraft->subfleet_id, $flight_subfleet_ids, true)) {
                        return back()->with('error', 'Selected aircraft is not allowed for this flight');
                    }
                }
            }

            $bid = app(\App\Services\BidService::class)->addBid($flight, $user, $aircraft);
            // If core settings prevented binding aircraft on creation, enforce it here
            if ($aircraft && empty($bid->aircraft_id)) {
                $bid->aircraft_id = $aircraft->id;
                $bid->save();
            }
        } catch (\Throwable $e) {
            return back()->with('error', 'Could not place bid: '.$e->getMessage());
        }

        return redirect()->route('frontend.flights.bids')->with('success', 'Bid placed');
    }
}

