<?php

namespace Modules\AAdvantageMiles\Http\Controllers;

use App\Contracts\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Modules\AAdvantageMiles\Models\MilesHistory;
use Modules\AAdvantageMiles\Models\ShopCategory;
use Modules\AAdvantageMiles\Models\ShopOrder;
use Modules\AAdvantageMiles\Models\ShopProduct;

class ShopController extends Controller
{
    public function index()
    {
        $cats = ShopCategory::where('is_active', 1)->orderBy('sort')->get();
        $products = ShopProduct::where('is_active', 1)->orderBy('sort')->get()->groupBy('category_id');
        return view('AAMiles::frontend.shop', compact('cats', 'products'));
    }

    public function purchases()
    {
        $orders = ShopOrder::where('user_id', Auth::id())->orderByDesc('created_at')->paginate(25);
        return view('AAMiles::frontend.purchases', compact('orders'));
    }

    public function buy(Request $request, $productId)
    {
        $user = Auth::user();
        $product = ShopProduct::where('is_active', 1)->findOrFail($productId);

        // Calculate available miles
        $totals = MilesHistory::selectRaw('SUM(miles) as total')->where('user_id', $user->id)->first();
        $balance = (int)($totals->total ?? 0);

        if ($product->cost_miles < 1) {
            return back()->with('error', 'Invalid product');
        }
        if ($balance < $product->cost_miles) {
            return back()->with('error', 'Not enough miles');
        }

        // Enforce per-user limit
        if (!empty($product->per_user_limit) && $product->per_user_limit > 0) {
            $count = ShopOrder::where('user_id', $user->id)->where('product_id', $product->id)->count();
            if ($count >= $product->per_user_limit) {
                return back()->with('error', 'Purchase limit reached for this item');
            }
        }

        // Enforce stock limit (global)
        if (!empty($product->stock_limit) && $product->stock_limit > 0) {
            $sold = ShopOrder::where('product_id', $product->id)->count();
            if ($sold >= $product->stock_limit) {
                return back()->with('error', 'This item is out of stock');
            }
        }

        DB::transaction(function () use ($user, $product) {
            // Deduct miles
            MilesHistory::create([
                'user_id' => $user->id,
                'issued_by' => 0,
                'source' => 'shop',
                'event_id' => (string)$product->id,
                'note' => 'Shop Purchase: '.$product->name,
                'miles' => -1 * abs((int)$product->cost_miles),
                'meta' => [],
                'created_at' => now(),
            ]);

            // Record purchase
            $order = ShopOrder::create([
                'user_id' => $user->id,
                'product_id' => $product->id,
                'cost_miles' => (int)$product->cost_miles,
                'meta' => [],
                'created_at' => now(),
            ]);

            // If this product is a Flight Pass, log it to adv_flight_passes
            $pmeta = is_array($product->meta ?? null) ? $product->meta : [];
            $is_pass = (($pmeta['type'] ?? null) === 'flight_pass');
            $flight_id = (string)($pmeta['flight_id'] ?? '');
            if ($is_pass && $flight_id !== '') {
                // If the product meta carries a pre-bound aircraft_id, persist it on the pass
                $aircraft_id = (int)($pmeta['aircraft_id'] ?? 0) ?: null;
                \Modules\AAdvantageMiles\Models\FlightPass::create([
                    'user_id'    => $user->id,
                    'flight_id'  => $flight_id,
                    'aircraft_id'=> $aircraft_id,
                    'order_id'   => $order->id,
                    'status'     => 'active',
                    'created_at' => now(),
                    'expires_at' => now()->addDays(30),
                ]);
            }
        });

        return redirect()->route('AAMiles.purchases')->with('success', 'Purchased: '.$product->name);
    }

    // Allow user to delete a purchase they made; refund the miles only if the pass/item hasn't been used
    public function deletePurchase(Request $request, int $orderId)
    {
        $userId = Auth::id();
        $order = ShopOrder::where('id', $orderId)->where('user_id', $userId)->firstOrFail();
        $product = ShopProduct::find($order->product_id);

        // Determine if it was a Flight Pass and whether it was used
        $meta = is_array($order->meta ?? null) ? $order->meta : [];
        $pmeta = is_array($product->meta ?? null) ? $product->meta : [];
        $is_pass = (($pmeta['type'] ?? null) === 'flight_pass');
        $was_used = $is_pass && !empty($meta['used']);

        DB::transaction(function () use ($order, $was_used) {
            // Refund miles only if NOT used
            if (!$was_used) {
                MilesHistory::create([
                    'user_id' => $order->user_id,
                    'issued_by' => 0,
                    'source' => 'shop-refund',
                    'event_id' => (string)$order->product_id,
                    'note' => 'Refund: Purchase deleted',
                    'miles' => (int)$order->cost_miles,
                    'meta' => ['refund_of' => $order->id],
                    'created_at' => now(),
                ]);
            }

            // Remove any associated FlightPass row
            $fp = \Modules\AAdvantageMiles\Models\FlightPass::where('order_id', $order->id)->first();
            if ($fp) { $fp->delete(); }

            // Delete order
            $order->delete();
        });

        $msg = $was_used ? 'Purchase deleted (no refund since it was used)' : 'Purchase deleted (miles refunded)';
        return redirect(url('aadvantage-miles/purchases'))->with('success', $msg);
    }
}

