<?php

namespace Modules\AAdvantageMiles\Listeners;

use App\Events\PirepFiled;
use Modules\AAdvantageMiles\Models\MilesSettings;
use Modules\AAdvantageMiles\Services\MilesLedger;

class PirepFiledListener
{
    public function handle(PirepFiled $event): void
    {
        $settings = MilesSettings::first();
        if (!$settings || !$settings->enabled) {
            return;
        }

        $p = $event->pirep;
        $user_id = $p->user_id;
        $ledger = app(MilesLedger::class);

        // Mark Flight Pass used if applicable
        try {
            if (class_exists('\\Modules\\AAdvantageMiles\\Services\\FlightPassService') && $p->flight && $p->user) {
                $fps = new \Modules\AAdvantageMiles\Services\FlightPassService();
                $fps->markUsed($p->user, $p->flight->id, $p->id);
            }
        } catch (\Throwable $e) { /* ignore */ }

        // Minutes: flight_time preferred, fallback to block_time
        $minutes = (int)($p->flight_time ?: $p->block_time ?: 0);

        // Miles per minute
        if ($settings->miles_per_min_enabled && $settings->miles_per_min > 0 && $minutes > 0) {
            $miles = (int) round($minutes * $settings->miles_per_min);
            $ledger->add($user_id, $miles, 'pirep_time', (string)$p->id, "Completed Flight {$p->flight_number}", null, null, "pirep_time:{$p->id}");
        }

        // Smooth landing
        if ($settings->smooth_enabled && $p->landing_rate < 0 && $p->landing_rate > $settings->smooth_limit) {
            if ($settings->smooth_miles > 0) {
                $ledger->add($user_id, (int)$settings->smooth_miles, 'smooth_landing', (string)$p->id, "Smooth landing {$p->landing_rate} fpm", null, null, "smooth_landing:{$p->id}");
            }
        }

        // Bad landing
        if ($settings->bad_enabled && $p->landing_rate < $settings->bad_limit) {
            if ($settings->bad_miles > 0) {
                $ledger->add($user_id, (int)(-1 * abs($settings->bad_miles)), 'bad_landing', (string)$p->id, "Bad landing {$p->landing_rate} fpm", null, null, "bad_landing:{$p->id}");
            }
        }

        // Score bonus: 1 mile per point, cap score_max; only if score > 1
        if ($settings->score_bonus_enabled) {
            $score = (int)($p->score ?? 0);
            if ($score > 1) {
                $bonus = min($score, (int)$settings->score_max);
                if ($bonus > 0) {
                    $ledger->add($user_id, $bonus, 'pirep_score_bonus', (string)$p->id, 'PIREP score bonus', ['score' => $score], null, "pirep_score_bonus:{$p->id}");
                }
            }
        }

        // Airport miles with per-rule condition (departure/arrival/both)
        $dep = strtoupper($p->dpt_airport_id ?? '');
        $arr = strtoupper($p->arr_airport_id ?? '');
        $rules = $settings->airport_miles ?? [];
        foreach ($rules as $rule) {
            $icao = strtoupper($rule['icao'] ?? '');
            $miles = (int)($rule['miles'] ?? 0);
            $when = strtolower($rule['when'] ?? 'both');
            if (!$icao || $miles <= 0) { continue; }
            $award_dep = in_array($when, ['departure','both'], true) && $dep === $icao;
            $award_arr = in_array($when, ['arrival','both'], true) && $arr === $icao;
            if ($award_dep) {
                $ledger->add($user_id, $miles, 'airport_bonus', (string)$p->id, "Airport bonus {$icao} (Departure)", null, null, "airport_bonus:{$p->id}:{$icao}:dep");
            }
            if ($award_arr) {
                $ledger->add($user_id, $miles, 'airport_bonus', (string)$p->id, "Airport bonus {$icao} (Arrival)", null, null, "airport_bonus:{$p->id}:{$icao}:arr");
            }
        }
    }
}

