<?php

namespace Modules\TripPlannerAdvanced\Http\Controllers\Frontend;

use App\Contracts\Controller;
use App\Models\Aircraft;
use App\Models\Enums\AircraftState;
use App\Models\Enums\AircraftStatus;
use App\Repositories\FlightRepository;
use App\Services\UserService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class FlightHelperController extends Controller
{
    public function __construct(private readonly FlightRepository $flightRepo, private readonly UserService $userSvc) {}

    /**
     * Return aircraft list for a flight limited to intersection of user-allowed subfleets
     * and the subfleets assigned to this flight. This is module-scoped and avoids
     * modifying core endpoints.
     */
    public function aircraft(string $id, Request $request)
    {
        $flight = $this->flightRepo->with('subfleets')->find($id);
        if (!$flight) {
            return response()->json([]);
        }

        $user_subfleets = $this->userSvc->getAllowableSubfleets(Auth::user())->pluck('id')->toArray();
        $flight_subfleets = $flight->subfleets->pluck('id')->toArray();
        $subfleet_ids = !empty($flight_subfleets) ? array_intersect($user_subfleets, $flight_subfleets) : $user_subfleets;

        $where = [
            'state' => AircraftState::PARKED,
            'status' => AircraftStatus::ACTIVE,
        ];

        // Respect phpVMS "only aircraft at dpt" unless TPA decides to bypass later via a module setting
        if (setting('pireps.only_aircraft_at_dpt_airport')) {
            $where['airport_id'] = $flight->dpt_airport_id;
        }

        $aircraft = Aircraft::where($where)
            ->whereIn('subfleet_id', $subfleet_ids)
            ->orderBy('icao')
            ->orderBy('registration')
            ->get(['id', 'icao', 'registration', 'name']);

        // Map to a compact format used by the modal select
        return $aircraft->map(function ($ac) {
            return [
                'id' => $ac->id,
                'icao' => $ac->icao,
                'registration' => $ac->registration,
                'name' => $ac->name,
            ];
        });
    }
}

