<?php

namespace Modules\TripPlannerAdvanced\Providers;

use Illuminate\Routing\Router;
use Illuminate\Foundation\Support\Providers\RouteServiceProvider as ServiceProvider;
use Illuminate\Support\Facades\Route;
use Modules\TripPlannerAdvanced\Http\Controllers\Admin\TripsAdminController;
use Modules\TripPlannerAdvanced\Http\Controllers\Admin\TripTemplatesAdminController;
use Modules\TripPlannerAdvanced\Http\Controllers\Admin\SettingsController;
use Modules\TripPlannerAdvanced\Http\Controllers\Frontend\IndexController;

/**
 * Register the routes required for your module here
 */
class RouteServiceProvider extends ServiceProvider
{
    /**
     * The root namespace to assume when generating URLs to actions.
     *
     * @var string
     */
    protected $namespace = 'Modules\TripPlannerAdvanced\Http\Controllers';

    /**
     * Called before routes are registered.
     *
     * Register any model bindings or pattern based filters.
     *
     * @param  Router $router
     * @return void
     */
    public function before(Router $router)
    {
        //
    }

    /**
     * Define the routes for the application.
     *
     * @param \Illuminate\Routing\Router $router
     *
     * @return void
     */
    public function boot()
    {
        parent::boot();
        $this->registerWebRoutes();
        $this->registerAdminRoutes();
    }

    /**
     *
     */
    protected function registerWebRoutes(): void
    {
        $config = [
            'as'         => 'tripplanner.',
            'namespace'  => $this->namespace.'\Frontend',
            'middleware' => ['web'],
        ];

        Route::group($config, function () {
            // Test route to verify module loading
            // Optional legacy route tester (if enabled in config)
            if (config('tripplanneradvanced.enable_legacy_trips_routes', false)) {
                Route::get('/trips/test-routes', function() {
                    return response()->json([
                        'message' => 'Trip Planner legacy routes are working',
                        'available_routes' => [
                            'tripplanner.index' => route('tripplanner.index'),
                            'tripplanner.create' => route('tripplanner.create'),
                            'tripplanner.destroy' => 'DELETE /trip-planner/{trip}'
                        ]
                    ]);
                });
            }

            // Primary routes under /trip-planner
            Route::group(['middleware' => ['auth'], 'prefix' => 'trip-planner'], function () {
                Route::get('/', [IndexController::class, 'index'])->name('index');
                Route::get('/create', [IndexController::class, 'create'])->name('create');
                Route::post('/', [IndexController::class, 'store'])->name('store');
                Route::get('/{trip}', [IndexController::class, 'show'])->name('show');
                Route::delete('/{trip}', [IndexController::class, 'destroy'])->name('destroy');

                // Module-contained endpoint to add a bid and (optionally) auto-reposition
                Route::post('/bid/add', [\Modules\TripPlannerAdvanced\Http\Controllers\Frontend\BidHookController::class, 'add'])
                    ->name('bid.add');

                // Trip Planner-scoped PIREP create which filters aircraft by the trip flight subfleets
                Route::get('/pireps/create', [\Modules\TripPlannerAdvanced\Http\Controllers\Frontend\PirepProxyController::class, 'create'])
                    ->name('pireps.create');
            });

                    // Aircraft list for a given flight (module-scoped; used by bid modal)
                    Route::get('/flights/{id}/aircraft', [\Modules\TripPlannerAdvanced\Http\Controllers\Frontend\FlightHelperController::class, 'aircraft'])
                        ->name('flights.aircraft');

            // Alias expected by phpVMS module listing: /tripplanneradvanced → index
            Route::get('/tripplanneradvanced', [IndexController::class, 'index'])->middleware(['auth']);

            // Optional legacy aliases under /trips
            if (config('tripplanneradvanced.enable_legacy_trips_routes', false)) {
                Route::group(['middleware' => ['auth'], 'prefix' => 'trips'], function () {
                    Route::get('/', [IndexController::class, 'index']);
                    Route::get('/create', [IndexController::class, 'create']);
                    Route::post('/', [IndexController::class, 'store']);
                    Route::get('/{trip}', [IndexController::class, 'show']);
                    Route::delete('/{trip}', [IndexController::class, 'destroy']);
                });
            }
        });
    }

    protected function registerAdminRoutes(): void
    {
        $config = [
            'as'         => 'admin.tripplanner.',
            'prefix'     => 'admin/trip-planner',
            'namespace'  => $this->namespace.'\Admin',
            'middleware' => ['web', 'auth', 'ability:admin,admin-access'],
        ];

        Route::group($config, function () {
            Route::get('/', [TripsAdminController::class, 'index'])->name('index');
            Route::get('/create', [TripsAdminController::class, 'create'])->name('create');
            Route::post('/', [TripsAdminController::class, 'store'])->name('store');

            // Settings page
            Route::get('/settings', [SettingsController::class, 'index'])->name('settings');
            Route::post('/settings', [SettingsController::class, 'save'])->name('settings.save');

            Route::group(['prefix' => '/templates', 'as' => 'templates.'], function () {
                Route::get('/', [TripTemplatesAdminController::class, 'index'])->name('index');
                Route::post('/', [TripTemplatesAdminController::class, 'store'])->name('store');
                Route::get('/create', [TripTemplatesAdminController::class, 'create'])->name('create');
                Route::get('/{id}', [TripTemplatesAdminController::class, 'show'])->name('show');
                Route::get('/{id}/edit', [TripTemplatesAdminController::class, 'edit'])->name('edit');
                Route::patch('/{id}', [TripTemplatesAdminController::class, 'update'])->name('update');
                Route::delete('/{id}', [TripTemplatesAdminController::class, 'destroy'])->name('destroy');
            });
            Route::get('/{id}', [TripsAdminController::class, 'show'])->name('show');
            Route::get('/{id}/edit', [TripsAdminController::class, 'edit'])->name('edit');
        });

        // Alias expected by phpVMS module listing: /admin/tripplanneradvanced → admin index/settings
        Route::group([
            'as' => 'admin.tripplanner.alias.',
            'prefix' => 'admin',
            'namespace' => $this->namespace.'\Admin',
            'middleware' => ['web', 'auth', 'ability:admin,admin-access'],
        ], function () {
            Route::get('tripplanneradvanced', [SettingsController::class, 'index']);
        });
    }
}
