<?php

namespace Modules\TripPlannerAdvanced\Http\Controllers\Frontend;

use App\Contracts\Controller;
use App\Models\Aircraft;
use App\Models\Enums\AircraftState;
use App\Models\Enums\AircraftStatus;
use App\Models\Enums\PirepFieldSource;
use App\Models\Enums\PirepSource;
use App\Models\Fare;
use App\Models\Pirep;
use App\Models\Subfleet;
use Illuminate\Contracts\View\View;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PirepProxyController extends Controller
{
    /**
     * Trip Planner-scoped PIREP create that filters aircraft to the trip flight's subfleets.
     * Returns the core "pireps.create" view with adjusted variables.
     */
    public function create(Request $request): View
    {
        $user = Auth::user();

        // Resolve needed services/repositories from the container (no core edits)
        $flightRepo = app('App\\Repositories\\FlightRepository');
        $airlineRepo = app('App\\Repositories\\AirlineRepository');
        $pirepFieldRepo = app('App\\Repositories\\PirepFieldRepository');
        $userSvc = app('App\\Services\\UserService');

        $pirep = null;
        $fare_values = [];
        $simbrief = null;
        $simbrief_id = null;
        $aircraft = null;
        $aircraft_list = [];

        // If a flight_id is present, pre-populate the PIREP from the flight
        $flight = null;
        if ($request->has('flight_id')) {
            $flight = $flightRepo->find($request->input('flight_id'));
            $pirep = Pirep::fromFlight($flight);
        }

        // If SimBrief ID provided, mimic core behavior for preselecting aircraft/fares
        if ($request->has('sb_id')) {
            $simbrief_id = $request->input('sb_id');
            $simbrief = \App\Models\SimBrief::find($simbrief_id);
            $pirep = Pirep::fromSimBrief($simbrief);

            $aircraft = $simbrief->aircraft;
            $aircraft_list[$aircraft->subfleet->name] = [];
            $aircraft_list[$aircraft->subfleet->name][$aircraft->id] = $aircraft->name.' - '.$aircraft->registration;

            if (!empty($simbrief->fare_data)) {
                $fare_values = json_decode($simbrief->fare_data, true);
                $fares = [];
                $fare_data = json_decode($simbrief->fare_data, true);
                foreach ($fare_data as $fare) {
                    $fares[] = new Fare($fare);
                }
                $aircraft->subfleet->fares = collect($fares);
            }
        } else {
            // Build aircraft list filtered by the flight's subfleets and the user's allowable subfleets
            // Fallback to user's allowable subfleets if no flight given
            $user_subfleet_ids = $userSvc->getAllowableSubfleets($user)->pluck('id')->toArray();

            $flight_subfleet_ids = [];
            if ($flight) {
                $flight_subfleet_ids = $flight->subfleets->pluck('id')->toArray();
                if ((blank($flight_subfleet_ids) || count($flight_subfleet_ids) === 0) && setting('flights.only_company_aircraft', false)) {
                    $flight_subfleet_ids = Subfleet::where(['airline_id' => $flight->airline_id])->pluck('id')->toArray();
                }
            }

            $subfleet_ids = filled($flight_subfleet_ids)
                ? array_intersect($user_subfleet_ids, $flight_subfleet_ids)
                : $user_subfleet_ids;

            // Prepare aircraft query with common restrictions
            $where = [
                'state' => AircraftState::PARKED,
                'status' => AircraftStatus::ACTIVE,
            ];

            if ($flight && setting('pireps.only_aircraft_at_dpt_airport')) {
                $where['airport_id'] = $flight->dpt_airport_id;
            }

            $aircraft_col = Aircraft::with(['subfleet'])
                ->where($where)
                ->whereIn('subfleet_id', $subfleet_ids)
                ->orderby('icao')->orderby('registration')
                ->get();

            // Optionally preselect aircraft_id passed via query if it's in the list
            $requested_ac_id = $request->input('aircraft_id');
            if ($requested_ac_id && $aircraft_col->pluck('id')->contains((int) $requested_ac_id)) {
                if (!$pirep) { $pirep = new Pirep(); }
                $pirep->aircraft_id = (int) $requested_ac_id;
            }

            // Convert to the grouped structure the view expects
            $aircraft_list[''] = '';
            foreach ($aircraft_col as $ac) {
                $group = optional($ac->subfleet)->type ?? 'Aircraft';
                if (!isset($aircraft_list[$group])) {
                    $aircraft_list[$group] = [];
                }
                $aircraft_list[$group][$ac->id] = $ac->name.' - '.$ac->registration;
            }
        }

        $pirep_source = filled(optional($pirep)->source) ? $pirep->source : PirepSource::MANUAL;

        return view('pireps.create', [
            'aircraft'      => $aircraft,
            'pirep'         => $pirep,
            'read_only'     => false,
            'airline_list'  => $airlineRepo->selectBoxList(true),
            'aircraft_list' => $aircraft_list,
            'airport_list'  => [],
            'pirep_fields'  => $pirepFieldRepo->whereIn('pirep_source', [$pirep_source, PirepFieldSource::BOTH])->get(),
            'field_values'  => [],
            'fare_values'   => $fare_values,
            'simbrief_id'   => $simbrief_id,
            'simbrief'      => $simbrief,
        ]);
    }
}

