<?php

namespace Modules\TripPlannerAdvanced\Http\Controllers\Frontend\Traits;

use App\Models\Bid;
use App\Models\Flight;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Modules\TripPlannerAdvanced\Models\TpaSettings;
use Modules\TripPlannerAdvanced\Models\FlightPirepTrip;

trait RepositionOnBid
{
    /**
     * If enabled in TPA settings, when the user places their first bid for a trip,
     * reposition their current location to the first leg's departure airport.
     * Safe-guards:
     * - Only if user has no bids on any flight in this trip yet
     * - Only if pilot is not already at the first departure
     */
    protected function maybeAutoReposition(User $user, Flight $flight): void
    {
        $settings = TpaSettings::first();
        if (!$settings || !$settings->auto_reposition_on_first_bid) {
            return;
        }

        // Identify the owning trip of this flight (module-owned flights set owner morph)
        if (!($flight->owner && $flight->owner instanceof \Modules\TripPlannerAdvanced\Models\TripReport)) {
            return; // not a TPA flight
        }
        $trip = $flight->owner; // TripReport

        // Find the first leg's departure
        $firstLeg = FlightPirepTrip::where('trip_report_id', $trip->id)->orderBy('order')->first();
        if (!$firstLeg) { return; }
        $firstFlight = Flight::find($firstLeg->flight_id);
        if (!$firstFlight) { return; }
        $firstDep = $firstFlight->dpt_airport_id;

        // Count the user's bids within this trip AFTER adding this bid
        // We only reposition when this is the FIRST bid placed for the trip
        $tripFlightIds = FlightPirepTrip::where('trip_report_id', $trip->id)->pluck('flight_id');
        $bidCount = Bid::where('user_id', $user->id)
            ->whereIn('flight_id', $tripFlightIds)
            ->count();

        if ($bidCount !== 1) { return; }

        // If not already at the first departure, reposition
        if ($user->curr_airport_id !== $firstDep) {
            $user->curr_airport_id = $firstDep;
            $user->save();
        }
    }
}

