<?php

namespace Modules\TripPlannerAdvanced\Http\Controllers\Frontend;

use App\Contracts\Controller;
use App\Models\Aircraft;
use App\Models\Subfleet;
use App\Models\Bid;
use App\Models\Enums\FlightType;
use App\Models\Flight;
use App\Models\Pirep;
use App\Repositories\AirlineRepository;
use App\Services\BidService;
use App\Services\FlightService;
use App\Services\GeoService;
use App\Services\ModuleService;
use App\Services\UserService;
use App\Services\AirportService;
use App\Exceptions\DuplicateFlight;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\Paginator;
use Modules\TripPlannerAdvanced\Models\FlightPirepTrip;
use Modules\TripPlannerAdvanced\Models\TripReport;
use Modules\TripPlannerAdvanced\Models\Enums\TripState;
use Modules\TripPlannerAdvanced\Models\TpaSettings;

/**
 * Class $CLASS$
 * @package
 */
class IndexController extends Controller
{
    public function __construct(
        public FlightService $flightService,
        public BidService $bidService,
        public AirlineRepository $airlineRepo,
        public ModuleService $moduleSvc,
        public GeoService $geoSvc,
        public UserService $userSvc,
        public AirportService $airportSvc
    ) {
    }

    /**
     * Get aircraft list filtered by user's rank (similar to PirepController)
     */
    private function getAircraftList($user, bool $add_blank = true): array
    {
        $aircraft = [];
        $subfleets = $this->userSvc->getAllowableSubfleets($user);

        if ($add_blank) {
            $aircraft[''] = '-- Select Aircraft --';
        }

        $subfleets->loadMissing('aircraft');

        foreach ($subfleets as $subfleet) {
            $tmp = [];
            foreach ($subfleet->aircraft as $ac) {
                $tmp[$ac->id] = $ac['name'].' - '.$ac['registration'];
            }

            if (!empty($tmp)) {
                $aircraft[$subfleet->type] = $tmp;
            }
        }

        return $aircraft;
    }

    /**
     * Filter subfleets by aircraft manufacturer
     */
    private function filterSubfleetsByManufacturer($subfleets, $manufacturer)
    {
        return $subfleets->filter(function ($subfleet) use ($manufacturer) {
            $aircraftType = strtolower($subfleet->type);

            switch ($manufacturer) {
                case 'boeing':
                    return str_contains($aircraftType, 'b737') ||
                           str_contains($aircraftType, 'b747') ||
                           str_contains($aircraftType, 'b757') ||
                           str_contains($aircraftType, 'b767') ||
                           str_contains($aircraftType, 'b777') ||
                           str_contains($aircraftType, 'b787') ||
                           str_contains($aircraftType, 'boeing') ||
                           str_contains($aircraftType, '737') ||
                           str_contains($aircraftType, '747') ||
                           str_contains($aircraftType, '757') ||
                           str_contains($aircraftType, '767') ||
                           str_contains($aircraftType, '777') ||
                           str_contains($aircraftType, '787');

                case 'airbus':
                    return str_contains($aircraftType, 'a319') ||
                           str_contains($aircraftType, 'a320') ||
                           str_contains($aircraftType, 'a321') ||
                           str_contains($aircraftType, 'a330') ||
                           str_contains($aircraftType, 'a340') ||
                           str_contains($aircraftType, 'a350') ||
                           str_contains($aircraftType, 'a380') ||
                           str_contains($aircraftType, 'airbus') ||
                           str_contains($aircraftType, '319') ||
                           str_contains($aircraftType, '320') ||
                           str_contains($aircraftType, '321') ||
                           str_contains($aircraftType, '330') ||
                           str_contains($aircraftType, '340') ||
                           str_contains($aircraftType, '350') ||
                           str_contains($aircraftType, '380');

                case 'bombardier':
                    return str_contains($aircraftType, 'crj') ||
                           str_contains($aircraftType, 'dash') ||
                           str_contains($aircraftType, 'dh8') ||
                           str_contains($aircraftType, 'dhc') ||
                           str_contains($aircraftType, 'q400') ||
                           str_contains($aircraftType, 'bombardier') ||
                           str_contains($aircraftType, 'canadair');

                case 'embraer':
                    return str_contains($aircraftType, 'erj') ||
                           str_contains($aircraftType, 'emb') ||
                           str_contains($aircraftType, 'e170') ||
                           str_contains($aircraftType, 'e175') ||
                           str_contains($aircraftType, 'e190') ||
                           str_contains($aircraftType, 'e195') ||
                           str_contains($aircraftType, 'embraer');

                case 'mcdonnell':
                    return str_contains($aircraftType, 'md') ||
                           str_contains($aircraftType, 'dc') ||
                           str_contains($aircraftType, 'mcdonnell') ||
                           str_contains($aircraftType, 'douglas');

                default:
                    return false;
            }
        });
    }

    /**
     * Filter subfleets by the new Equipment Category (single-select group).
     * Categories mirror Monthly Flight Assignments groupings.
     */
    private function filterSubfleetsByCategory($subfleets, string $category)
    {
        return $subfleets->filter(function ($subfleet) use ($category) {
            $hay = strtoupper(($subfleet->name ?? '').' '.($subfleet->type ?? ''));
            switch ($category) {
                case 'airbus_narrow':
                    return (bool) preg_match('/A3(19|20|21)/i', $hay);
                case 'boeing_narrow':
                    return (bool) preg_match('/B?(737|757)/i', $hay);
                case 'airbus_wide':
                    return (bool) preg_match('/A33(0)|A330|A34(0)|A350|A380/i', $hay);
                case 'boeing_wide':
                    return (bool) preg_match('/B?(767|777|787)/i', $hay);
                case 'regional':
                    return (bool) preg_match('/(CRJ|ERJ|E170|E175|E190|E195|EMB)/i', $hay);
                case 'mcdonnell':
                    return (bool) preg_match('/\b(MD|DC)\d+/i', $hay);
                default:
                    return false;
            }
        });
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return mixed
     */
    public function index(Request $request)
    {
        $trips = TripReport::whereHas('users', function ($q) {
            $q->where('user_id', Auth::user()->id);
        })->get();
        foreach($trips as $trip) {
            $completed = 0;
            $fpts = $trip->fpts()->orderBy('order')->get();
            foreach ($fpts as $fpt) {
                if ($fpt->pirep_id === null) {
                    //dd(Flight::find($fpt->flight_id));
                } else {
                    $completed++;
                }
            }

            // Prevent division by zero error
            $fpts_count = count($fpts);
            $prog = $fpts_count > 0 ? round($completed / $fpts_count * 100) : 0;

            $trip->progress = "{$completed}/{$trip->fpts->count()} ({$prog}%)";
        }
        return view('tpa::index', ['trips' => $trips]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param Request $request
     *
     * @return mixed
     */
    public function create(Request $request)
    {
        $user = Auth::user();

        // Get user's current location (from last pirep) or home airport as fallback
        $currentLocation = null;
        if ($user->curr_airport_id) {
            $currentLocation = $user->current_airport;
        } elseif ($user->home_airport_id) {
            $currentLocation = $user->home_airport;
        }

        // Honor ONLY the Trip Planner toggle for this module (ignore core phpVMS restriction here)
        $enforceRank = $this->tpaSetting('enforce_rank', false);
        $allowedCategories = [];
        if ($enforceRank) {
            // Use core logic to determine what the pilot can actually fly
            $userAllowedSubfleets = $this->userSvc->getAllowableSubfleets($user);

            $cats = ['airbus_narrow','boeing_narrow','airbus_wide','boeing_wide','regional','mcdonnell'];
            foreach ($cats as $c) {
                $allowedCategories[$c] = $this->filterSubfleetsByCategory($userAllowedSubfleets, $c)->isNotEmpty();
            }
        }

        return view('tpa::create', [
            'airlines' => $this->airlineRepo->selectBoxList(true),
            'aircraft' => $this->getAircraftList($user),
            'currentLocation' => $currentLocation,
            'lockToCurrent' => $this->tpaSetting('lock_to_current_location', true),
            'enforceRank' => $enforceRank,
            'allowedCategories' => $allowedCategories,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     *
     * @return mixed
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        // Check if user has a current location
        $userCurrentLocation = $user->curr_airport_id ?: $user->home_airport_id;
        if (!$userCurrentLocation) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'You must have a current location to create trips. Please complete a flight first or contact an administrator.');
        }

        // Validate the basic requirements
        $request->validate([
            'airports' => 'required|array|min:2',
            'airline_id' => 'required|exists:airlines,id',
            'equipment_category' => 'required|in:airbus_narrow,boeing_narrow,airbus_wide,boeing_wide,regional,mcdonnell'
        ]);

        // Validate that the departure airport matches user's current location (if enabled)
        if ($this->tpaSetting('lock_to_current_location', true)) {
            $userCurrentLocation = $user->curr_airport_id ?: $user->home_airport_id;
            $departureAirport = $request->input('airports')[0];

            if ($userCurrentLocation && $departureAirport !== $userCurrentLocation) {
                $currentAirportName = $user->current_airport ? $user->current_airport->name : ($user->home_airport ? $user->home_airport->name : 'Unknown');
                return redirect()->back()
                    ->withInput()
                    ->with('error', "You must depart from your current location: {$currentAirportName} ({$userCurrentLocation}). Please update the departure airport.");
            }
        }

        // Verify user has access to aircraft if rank restrictions are enabled
        if ($this->tpaSetting('enforce_rank', false)) {
            $allowedSubfleets = $this->userSvc->getAllowableSubfleets($user);
            if ($allowedSubfleets->isEmpty()) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'You do not have the required rank to create trips.');
            }
        }

        // Get the request
        $data = $request->all();
        // Use CHTrips flight number range 7000–7999 (avoid 9000s)
        $data['flight_number'] = random_int(7000, 7999);
        // Tag all Trip Planner flights so they don't collide with scheduled flights
        if (empty($data['route_code'])) {
            $data['route_code'] = 'TPA';
        }
        $data['minutes'] = 0;
        $data['hours'] = 0;
        $data['active'] = true;
        $data['visible'] = false;

        $airports = $data['airports'];
        $flight_type = $data['flight_type'];
        $equipment_category = $data['equipment_category'] ?? null;

        unset($data['flight_type']);
        unset($data['airports']);
        unset($data['equipment_category']);

        // First, create the trip.
        $tr = new TripReport();


        if ($data['name'] == "") {
            $tr->name = "Free Flight: {$airports[0]}->{$airports[count($airports) - 1]}";
        } else {
            $tr->name = $data['name'];
        }

        if ($data['description'] != "") {
            $tr->description = $data['description'];
        }

        // Save trip basics
        $tr->state = TripState::UPCOMING;
        $tr->save();

        // Attach creator as owner on the pivot so we can show who created the trip
        $tr->users()->attach($user->id, ['owner' => true]);

        // Determine subfleets: respect rank if enabled, otherwise allow ALL subfleets
        $allowedSubfleets = $this->tpaSetting('enforce_rank', false)
            ? $this->userSvc->getAllowableSubfleets($user)
            : Subfleet::with(['aircraft', 'fares'])->get();

        // If a category is selected, filter subfleets accordingly
        $filteredSubfleets = $equipment_category ? $this->filterSubfleetsByCategory($allowedSubfleets, $equipment_category) : $allowedSubfleets;
        if ($filteredSubfleets->isEmpty() && $equipment_category) {
            $msg = $this->tpaSetting('enforce_rank', false)
                ? 'No aircraft in the selected category are available for your rank.'
                : 'No aircraft in the selected category were found.';
            flash()->error($msg);
            return redirect()->back()->withInput();
        }

        // Now, create each flight based on the params
        for ($i = 0; $i < count($airports) - 1; $i++) {
            $data['dpt_airport_id'] = $airports[$i];
            $data['arr_airport_id'] = $airports[$i + 1];

            $data['route_leg'] = $i + 1;
            $data['flight_type'] = $flight_type[$i];

            try {
                $flight = $this->flightService->createFlight($data);
            } catch (DuplicateFlight $e) {
                // If a duplicate is detected, increment the random flight number and retry once
                Log::warning('TPA duplicate flight detected; retrying with new number', ['route_leg' => $data['route_leg'], 'flight_number' => $data['flight_number']]);
                $data['flight_number'] = random_int(7000, 7999);
                $flight = $this->flightService->createFlight($data);
            }

            $flight->owner()->associate($tr);
            $flight->save();

            // Assign subfleets to this flight (all in the chosen category or all allowed if none chosen)
            foreach ($filteredSubfleets as $subfleet) {
                // Use syncWithoutDetaching with IDs to avoid duplicate pivot entries and type errors
                $flight->subfleets()->syncWithoutDetaching([$subfleet->id]);
            }

            // TODO: Refactor this later when we find out WHY this is not behaving
            FlightPirepTrip::create([
                'trip_report_id' => $tr->id,
                'flight_id'      => $flight->id,
                'order'          => $i + 1
            ]);
            // $tr->flights()->attach($flight, ['order' => $i + 1]);
        }

        // Finally, add the flight to that user's bids.
        return to_route('tripplanner.show', ['trip' => $tr->id]);
    }

    /**
     * Show the specified resource.
     *
     * @param Request $request
     *
     * @return mixed
     */
    public function show($trip, Request $request)
    {
        $trip_report = TripReport::find($trip);
        $upcoming = [];
        $completed = [];
        if ($trip_report === null) {
            abort(404, "Trip Report Not Found");
        }
        $fpts = $trip_report->fpts()->orderBy('order')->get();
        foreach ($fpts as $fpt) {
            if ($fpt->pirep_id === null) {
                //dd(Flight::find($fpt->flight_id));
                $upcoming[] = Flight::with(['dpt_airport', 'arr_airport', 'alt_airport', 'airline'])->find($fpt->flight_id);
            } else {
                $completed[] = Pirep::find($fpt->pirep_id);
            }
        }
        $saved_flights = [];
        $bids = Bid::where('user_id', Auth::id())->get();
        foreach ($bids as $bid) {
            if (!$bid->flight) {
                $bid->delete();
                continue;
            }

            $saved_flights[$bid->flight_id] = $bid->id;
        }
        //dd([$upcoming, $completed, $fpts]);
        // Prevent division by zero error
        $fpts_count = count($fpts);
        $progress = $fpts_count > 0 ? round(count($completed) / $fpts_count * 100) : 0;

        // Generate map features for the next flight (if available)
        $map_features = null;
        $next_flight = array_shift($upcoming);
        if ($next_flight) {
            $map_features = $this->geoSvc->flightGeoJson($next_flight);
        }

        // Convert completed array to a paginator so the shared pireps.table partial works
        $perPage = 15;
        $page = Paginator::resolveCurrentPage() ?: 1;
        $completedCollection = collect($completed);
        $pireps = new LengthAwarePaginator(
            $completedCollection->forPage($page, $perPage)->values(),
            $completedCollection->count(),
            $perPage,
            $page,
            [
                'path' => Paginator::resolveCurrentPath(),
                'query' => $request->query(),
            ]
        );

        return view('tpa::show', [
            'name'          => $trip_report->name,
            'description'   => $trip_report->description,
            'flight'        => $next_flight,
            'saved'         => $saved_flights,
            'progress'      => $progress,
            'upcoming'      => $upcoming,
            'pireps'        => $pireps,
            'user'          => Auth::user(),
            'legs'          => $fpts->count(),
            'simbrief'      => !empty(setting('simbrief.api_key')),
            'simbrief_bids' => setting('simbrief.only_bids'),
            'acars_plugin'  => $this->moduleSvc->isModuleActive('VMSAcars'),
            'map_features'  => $map_features,
            'lockToCurrent' => $this->tpaSetting('lock_to_current_location', true),
            'autoReposition'=> $this->tpaSetting('auto_reposition_on_first_bid', false),
            // For view logic: show the auto-reposition note only before any legs are completed
            'hasCompleted'  => count($completed) > 0,
            'completedCount'=> count($completed),
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Request $request
     *
     * @return mixed
     */
    public function edit(Request $request)
    {
        return view('tpa::create');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     */
    public function update(Request $request)
    {
    }

    private function tpaSetting(string $key, $default = null)
    {
        $s = TpaSettings::first();
        return $s && isset($s->$key) ? (bool)$s->$key : config('tripplanneradvanced.' . $key, $default);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param $trip
     * @param Request $request
     */
    public function destroy($trip, Request $request)
    {
        $user = Auth::user();
        $trip_report = TripReport::find($trip);

        if (!$trip_report) {
            return redirect()->route('tripplanner.index')->with('error', 'Trip not found.');
        }

        // Check if the user is associated with this trip
        if (!$trip_report->users()->where('user_id', $user->id)->exists()) {
            return redirect()->route('chtrips.index')->with('error', 'You are not authorized to delete this trip.');
        }

        // Delete associated flight-pirep-trip records
        FlightPirepTrip::where('trip_report_id', $trip_report->id)->delete();

        // Detach users from the trip
        $trip_report->users()->detach();

        // Delete the flights created for this trip (if they are custom flights)
        $fpts = $trip_report->fpts;
        foreach ($fpts as $fpt) {
            $flight = Flight::find($fpt->flight_id);
            if ($flight && $flight->owner_type === TripReport::class && $flight->owner_id === $trip_report->id) {
                // Remove any bids for this flight
                Bid::where('flight_id', $flight->id)->delete();
                // Delete the flight
                $flight->delete();
            }
        }

        // Finally, delete the trip report
        $trip_report->delete();

        return redirect()->route('tripplanner.index')->with('success', 'Trip deleted successfully.');
    }
}
