@extends('tpa::layouts.frontend')

@section('title', 'Trip Planner')

@section('content')
  {{ Form::open(['route' => 'tripplanner.store', 'method' => 'POST']) }}
  <div class="card shadow-lg border-0 mx-auto" style="max-width: 1200px;">
    <div class="card-header d-flex align-items-center justify-content-between">
      <h5 class="mb-0">Create New Trip</h5>
      <a href="{{ route('tripplanner.index') }}" class="btn btn-sm btn-outline-secondary">Back to Trips</a>
    </div>
    <div class="card-body">
      @if ($errors->any())
        <div class="alert alert-danger">
          <ul class="mb-0">
            @foreach ($errors->all() as $error)
              <li>{{ $error }}</li>
            @endforeach
          </ul>
        </div>
      @endif
      @if (session('error'))
        <div class="alert alert-danger">{{ session('error') }}</div>
      @endif

      <style>
        .section-title{font-weight:600;margin-bottom:.75rem;position:relative;padding-left:1rem}
        .section-title:before{content:'';width:.5rem;height:.5rem;border-radius:50%;background:#0d6efd;position:absolute;left:0;top:.6rem}
        .section-panel{background-color:var(--bs-body-bg);border:1px solid var(--bs-border-color);border-radius:.5rem;padding:1rem 1.25rem}
        .aircraft-category-selection { display:grid; grid-template-columns: repeat(2, minmax(0,1fr)); gap:1rem; }
        @media (max-width: 575.98px) { .aircraft-category-selection { grid-template-columns: 1fr; } }
        .aircraft-category-selection .form-check { padding:0; margin:0; }
        .aircraft-category-selection .form-check-input { position:absolute; opacity:0; width:0; height:0; }
        .aircraft-category-selection .category-btn { width:100%; min-height:90px; padding:1rem; display:flex; flex-direction:column; align-items:center; justify-content:center; border-radius:.375rem; border:1px solid var(--bs-border-color); transition:all .2s; color:#fff; font-weight:600; position:relative; }
        .aircraft-category-selection .category-btn::after { content:'\2713'; position:absolute; top:.5rem; right:.5rem; width:1.5rem; height:1.5rem; line-height:1.5rem; text-align:center; border-radius:50%; background:rgba(255,255,255,.2); display:none; }
        .aircraft-category-selection .form-check-input:checked + .form-check-label .category-btn { outline:2px solid #fff; box-shadow:0 0 0 .2rem rgba(13,110,253,.25); }
        .aircraft-category-selection .form-check-input:checked + .form-check-label .category-btn::after { display:block; }
      </style>

      <div class="row g-4">
        <div class="col-lg-5">
          <h5 class="section-title">Route Planning</h5>
          <div class="section-panel">
            <div class="mb-3">
              <label for="airline_id" class="form-label">@lang('common.airline')</label>
              {{ Form::select('airline_id', $airlines, null , ['id'=>'airline_id','class' => 'form-select', 'required' => true]) }}
            </div>

            <div class="mb-3">
              <label for="start_airport" class="form-label">Start Airport</label>
              <select id="start_airport" class="form-select airport_search" name="airports[]" placeholder="Type To Begin Search" style="width: 100%" required>
                @if(!empty($currentLocation) && ($lockToCurrent ?? true))
                  <option value="{{ $currentLocation->id }}" selected>{{ $currentLocation->description }}</option>
                @endif
              </select>
              @if($lockToCurrent ?? true)
                <div class="form-text">You must start from your current location.</div>
              @else
                <div class="form-text">Start typing to search any airport (departure not locked).</div>
              @endif
            </div>

            <div class="d-flex align-items-center justify-content-between mb-2">
              <label class="mb-0">Trip Legs</label>
              <a href="#" class="btn btn-sm btn-primary" id="btnAddAirport">+ Add Leg</a>
            </div>
            <div id="airport_container"></div>
          </div>
        </div>

        <div class="col-lg-7">
          <h5 class="section-title">Trip Details</h5>
          <div class="section-panel mb-3">
            <div class="row g-3 mt-1">
              <div class="col-sm-6">
                {{ Form::text('flight_number', old('flight_number'), [
                    'placeholder' => 'e.g., AA1234',
                    'class' => 'form-control',
                    'aria-label' => 'Flight Number (optional)'
                ]) }}
                <div class="form-text">Flight Number (optional)</div>
              </div>
              <div class="col-sm-6">
                {{ Form::text('route_code', old('route_code'), [
                    'placeholder' => __('pireps.codeoptional'),
                    'class' => 'form-control',
                    'aria-label' => 'Route Code (optional)'
                ]) }}
                <div class="form-text">Route Code (optional)</div>
              </div>
            </div>

            <div class="mt-3">
              <label class="form-label">Trip Name</label>
              <input type="text" class="form-control" name="name" value="{{ old('name') }}"/>
            </div>
            <div class="mt-3">
              <label class="form-label">Description</label>
              <textarea class="form-control" name="description" rows="4">{{ old('description') }}</textarea>
            </div>
          </div>

          <h5 class="section-title">Select Equipment</h5>
          <div class="section-panel">
            <div class="aircraft-category-selection">
              @php
                $categories = [
                  'airbus_narrow' => ['label' => 'Airbus Narrow Body', 'class' => 'btn-primary'],
                  'boeing_narrow' => ['label' => 'Boeing Narrow Body', 'class' => 'btn-success'],
                  'airbus_wide'   => ['label' => 'Airbus Wide Body',   'class' => 'btn-info'],
                  'boeing_wide'   => ['label' => 'Boeing Wide Body',   'class' => 'btn-warning'],
                  'regional'      => ['label' => 'Regional Jets',      'class' => 'btn-danger'],
                  'mcdonnell'     => ['label' => 'McDonnell/Other',    'class' => 'btn-dark'],
                ];
              @endphp

              @php
                $rankFilteredCategories = $categories;
                if(($enforceRank ?? false) && !empty($allowedCategories)) {
                  $rankFilteredCategories = array_filter($categories, function($v, $k) use ($allowedCategories){
                    return $allowedCategories[$k] ?? false;
                  }, ARRAY_FILTER_USE_BOTH);
                }
              @endphp

              @forelse($rankFilteredCategories as $key => $meta)
                <div class="form-check">
                  <input class="form-check-input" type="radio" name="equipment_category" id="cat_{{ $key }}" value="{{ $key }}">
                  <label class="form-check-label" for="cat_{{ $key }}">
                    <div class="btn category-btn {{ $meta['class'] }}">{{ $meta['label'] }}</div>
                  </label>
                </div>
              @empty
                <div class="alert alert-warning mb-0">No eligible equipment categories are available for your current rank.</div>
              @endforelse
            </div>
            <div class="form-text mt-2">Pick one category. Flight number and code are auto-assigned.</div>
          @error('equipment_category')
            <div class="text-danger small">{{ $message }}</div>
          @enderror
          </div>
        </div>
      </div>
    </div>
    <div class="card-footer d-flex justify-content-end">
      <a href="{{ route('tripplanner.index') }}" class="btn btn-secondary me-2">Cancel</a>
      <button type="submit" class="btn btn-primary" id="createTripBtn">Create Trip</button>
    </div>
  </div>
  {{ Form::close() }}

    </div>
@endsection

@section('scripts')
  <script>
    function delInput(id) {
      const el = document.getElementById(`airport-${id}`);
      if (el) el.remove();
    }

    function selectInput(id) {
      return `
<div class="my-2" id="airport-${id}">
  <div class="card">
    <div class="card-body">
      <div class="row g-3 align-items-end">
        <div class="col">
          <label class="form-label">Flight Type</label>
          <select class="form-select" name="flight_type[]">
            <option value="C">Passenger (Charter)</option>
            <option value="A">Cargo (Additional)</option>
            <option value="E">VIP Flight</option>
            <option value="G">Passenger (Additional)</option>
            <option value="H">Cargo (Charter)</option>
            <option value="I">Ambulance</option>
            <option value="K">Training</option>
            <option value="M">Mail Service</option>
            <option value="O">Passenger (Special Charter)</option>
            <option value="P">Positioning</option>
            <option value="T">Technical Test</option>
            <option value="W">Military</option>
            <option value="X">Technical Stop</option>
          </select>
        </div>
        <div class="col">
          <label class="form-label">Destination</label>
          <select class="form-select airport_search" name="airports[]" style="width: 100%" placeholder="Type to search"></select>
        </div>
      </div>
    </div>
    <a href="#" class="btn btn-danger btn-sm delete w-100" onclick="delInput(${id})">Remove Leg</a>
  </div>
`;
    }

    document.addEventListener('DOMContentLoaded', function() {
      function initAirportSelect(el) {
        if (el && !el.tomselect) {
          new TomSelect(el, {
            valueField: 'id',
            labelField: 'description',
            searchField: 'description',
            maxItems: 1,
            openOnFocus: false, // avoid auto-opening when focusing multiple fields, which causes flicker
            shouldLoad: function(query) { return query.length >= 2; }, // wait for 2+ chars to reduce chatter/flicker
            loadThrottle: 250,
            load: function(query, callback) {
              if (query.length < 2) { return callback(); }
              var url = new URL('{{ Config::get("app.url") }}/api/airports/search');
              var params = {
                search: query,
                hubs: el.classList.contains('hubs_only') ? 1 : 0,
                page: 1,
                orderBy: 'id',
                sortedBy: 'asc'
              };
              Object.keys(params).forEach(key => url.searchParams.append(key, params[key]));
              fetch(url)
                .then(response => response.json())
                .then(json => callback(json.data))
                .catch(() => callback());
            }

          });
        }
      }

      // Initialize any airport_search selects that may not be initialized (safety)
      document.querySelectorAll('select.airport_search').forEach(initAirportSelect);

      document.getElementById('btnAddAirport').addEventListener('click', function(e) {
        e.preventDefault();
        const id = Date.now();
        const container = document.getElementById('airport_container');
        container.insertAdjacentHTML('beforeend', selectInput(id));
        const newSelect = container.querySelector(`#airport-${id} select.airport_search`);
        initAirportSelect(newSelect);
      });
    });

      // Prevent double submit (outside of TomSelect init)
      const createBtn = document.getElementById('createTripBtn');
      if (createBtn) {
        createBtn.closest('form').addEventListener('submit', function(e) {
          createBtn.disabled = true;
          createBtn.innerText = 'Creating...';
        });
      }

  </script>
@endsection
