@extends('app')
@section('title', 'Destinations Map')

@section('content')
  <div class="card card-flush border-hover-primary mb-10">
    <div class="card-header align-items-center">
      <div class="d-flex flex-row gap-6 align-items-center w-100">
        <div>
          <div class="text-muted small">Origin</div>
          <div id="cc-origin" class="fw-bold text-primary" style="min-width:180px"></div>
        </div>
        <div>
          <div class="text-muted small">Destination</div>
          <div id="cc-destination" class="fw-bold text-primary" style="min-width:180px"></div>
        </div>
        <div class="ms-auto d-flex align-items-center gap-3">
          <div>
            <select id="cc-airline" class="form-select form-select-sm" style="min-width:200px"></select>
          </div>
          <div id="cc-flightcount" class="text-muted small"></div>
          <button id="cc-acars" class="btn btn-sm btn-primary">Toggle ACARS</button>
          <button id="cc-reset" class="btn btn-sm btn-secondary">Reset Map</button>
        </div>
      </div>
    </div>
    <div class="card-body p-0">
      <div id="map" style="width: 100%; height: {{ $map_height }}px"></div>
    </div>
  </div>

  <div id="cc-schedules" class="card card-flush border-hover-primary mb-10" style="display:none">
    <div class="card-header"><h3 class="card-title">Schedules</h3></div>
    <div class="card-body p-0">
      <div class="table-responsive">
        <table class="table align-middle table-row-bordered table-row-solid gy-4 gs-9" id="cc-schedules-table">
          <thead class="border-gray-800 text-gray-800 fs-6 fw-semibold bg-lighten">
            <tr>
              <th>Ident</th>
              <th>Airline</th>
              <th>Origin</th>
              <th>Destination</th>
              <th>Dep</th>
              <th>Arr</th>
              <th>Aircraft</th>
              <th>Details</th>
            </tr>
          </thead>
          <tbody></tbody>
        </table>
      </div>
    </div>
  </div>

  <div id="map-info-box" class="map-info-box" style="display:none"></div>
  <div id="live_flights" style="display:none"></div>
@endsection

@section('css')
  <style>
    /* Improve dark mode contrast */
    #cc-schedules .card-title,
    #live_flights .card-title { color: #fff !important; }
    #cc-schedules table thead th,
    #cc-schedules table tbody td,
    #live_flights table thead th,
    #live_flights table tbody td { color: #fff !important; }
    #cc-schedules .bg-lighten,
    #live_flights .bg-lighten { background-color: rgba(255,255,255,0.06) !important; }
    #map-info-box { color: #fff !important; }
  </style>
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" />
@endsection


@section('scripts')
  <link rel="stylesheet" href="https://unpkg.com/leaflet@1.7.1/dist/leaflet.css" />
  <script>
    (function(){
      const leaflet = window.L || window.leaflet || L;
      const map = phpvms.map.render_base_map({});
      // Layers
      const layerAirports = new leaflet.FeatureGroup().addTo(map);
      const layerDirect = new leaflet.FeatureGroup().addTo(map);
      const layerConnect = new leaflet.FeatureGroup().addTo(map);
      const layerAcars = new leaflet.FeatureGroup().addTo(map);
      let acarsTimer = null;
      let acarsOn = false;
      let currentOrigin = null; let currentDest = null; let airlineId = '';
      const onlyFromCurrent = {{ setting('pilots.only_flights_from_current') ? 'true' : 'false' }};
      const userCurr = "{{ auth()->user()->curr_airport_id }}";
      const bidsUrl = "{{ route('frontend.flights.bids') }}";
      const blockAircraft = {{ setting('bids.block_aircraft', false) ? 'true' : 'false' }};

      // Helpers
      function setTopBox(el, apt){
        const box = document.getElementById(el);
        if(!apt){ box.textContent=''; return; }
        box.innerHTML = `<div class=\"fs-5 fw-bold text-primary\">${apt.icao||apt.id||''}</div><div class=\"small text-muted\">${apt.name||''}</div>`;
      }
      function fetchJSON(url, params={}){
        const qs = new URLSearchParams(params).toString();
        return phpvms.request({ url: qs? `${url}?${qs}`: url }).then(r=>r.data.data||[]);
      }
      function canBidForFlight(flightId, dpt){
        if(onlyFromCurrent && dpt && String(dpt).toUpperCase() !== String(userCurr||'').toUpperCase()){
          return Promise.resolve(false);
        }
        if(!blockAircraft){
          // Match core flights page behavior: don't require available aircraft
          return Promise.resolve(true);
        }
        return phpvms.request({ url: `/api/flights/${flightId}/aircraft` })
          .then(resp => ((resp.data && resp.data.length!==undefined) ? resp.data : resp.data.data)||[])
          .then(arr => (arr||[]).length>0)
          .catch(()=>false);
      }
      function prepareBidButtons(){
        document.querySelectorAll('#cc-schedules-table button.cc-bid').forEach(btn=>{
          const fid = btn.getAttribute('data-flight-id');
          const dpt = btn.getAttribute('data-dpt');
          canBidForFlight(fid, dpt).then(ok=>{
            if(ok){
              btn.disabled = false;
              btn.addEventListener('click', ()=>{
                phpvms.bids.addBid(fid).then(()=>{ window.location = bidsUrl; })
                  .catch(err=>{ const msg=(err && err.response && err.response.data && err.response.data.details)||err.message||'Error adding bid'; alert(msg); });
              }, { once:true });
            }else{
              btn.disabled = true;
              btn.title = 'Not eligible to bid this flight';
            }
          });
        });
      }
      function drawGeodesic(a,b,color,width){
        if(!a||!b) return null;
        const g = new L.Geodesic([], { color: color||'green', weight: width||2, opacity: 1, wrap:false });
        g.setLatLngs([[a.lat,a.lon],[b.lat,b.lon]]); return g.addTo(color==='#1f4fa3'?layerConnect:layerDirect);
      }
      function clearAll(){ layerAirports.clearLayers(); layerDirect.clearLayers(); layerConnect.clearLayers(); setTopBox('cc-origin'); setTopBox('cc-destination'); currentOrigin=null; currentDest=null; document.getElementById('cc-schedules').style.display='none'; }

      // Airlines dropdown
      fetchJSON('/modules/destinations-map/airlines').then(list=>{
        const sel = document.getElementById('cc-airline');
        sel.innerHTML = list.map(a=>`<option value="${a.id}">${a.icao} — ${a.name}</option>`).join('');
        const defaultAal = list.find(a=>a.icao==='AAL') || list[0];
        if (defaultAal) { sel.value = String(defaultAal.id); airlineId = sel.value; }
        sel.addEventListener('change', ()=>{ airlineId = sel.value; loadOrigins(); if(currentOrigin){ onOriginClick(currentOrigin); } });
        loadOrigins();
      });

      function loadOrigins(){
        layerAirports.clearLayers(); layerDirect.clearLayers(); layerConnect.clearLayers();
        fetchJSON('/modules/destinations-map/origins', { airline_id: airlineId }).then(airports=>{
          airports.forEach(apt=>{
            const marker = leaflet.circleMarker([apt.lat, apt.lon], { radius: (apt.hub?5:3)||3, color:'black', fillColor:'black', fillOpacity:1 });
            marker.bindTooltip(`${apt.icao} - ${apt.name}`);
            marker.on('click', ()=>onOriginClick(apt));
            marker.addTo(layerAirports);
          });
          try{ map.fitBounds(layerAirports.getBounds()); }catch(e){}
        });
      }

      function onOriginClick(apt){
        currentOrigin = apt; currentDest=null; setTopBox('cc-origin', apt); setTopBox('cc-destination', null);
        layerDirect.clearLayers(); layerConnect.clearLayers();
        fetchJSON('/modules/destinations-map/destinations', { origin: apt.id, airline_id: airlineId }).then(dests=>{
          const pts=[leaflet.latLng(apt.lat,apt.lon)];
          dests.forEach(dst=>{
            const arc = drawGeodesic(apt, dst, 'green', 2);
            if(arc){ pts.push(leaflet.latLng(dst.lat,dst.lon)); }
            const dot = leaflet.circleMarker([dst.lat,dst.lon], {radius:3, color:'green', fillColor:'green', fillOpacity:1});
            dot.bindTooltip(`${dst.icao} - ${dst.name}`);
            dot.on('click', ()=>onDestinationClick(apt,dst));
            layerDirect.addLayer(dot);
          });
          try{ map.fitBounds(leaflet.featureGroup(pts.map(p=>leaflet.marker(p))).getBounds()); }catch(e){}
        });
      }

      function onDestinationClick(dep, arr){
        currentDest = arr; setTopBox('cc-destination', arr);
        layerConnect.clearLayers();
        drawGeodesic(dep, arr, 'green', 4);
        try{ map.fitBounds(leaflet.featureGroup([leaflet.marker([dep.lat,dep.lon]), leaflet.marker([arr.lat,arr.lon])]).getBounds(), {padding:[40,40]}); }catch(e){}
        // No fan-out on the second click; focus on the selected city pair only.
        layerConnect.clearLayers();
        // load schedules table for the primary destination
        fetchJSON('/modules/destinations-map/schedules', { origin: dep.id, destination: arr.id, airline_id: airlineId }).then(rows=>{
          const tbody = document.querySelector('#cc-schedules-table tbody');
          tbody.innerHTML = rows.map(r=>`<tr>
            <td>${r.airline_logo?`<img src="${r.airline_logo}" alt="${r.airline||''}" style="height:16px;vertical-align:middle"/>`:''} <span class="ms-1">${r.ident||''}</span></td>
            <td>${r.airline||''}</td>
            <td><i class="bi bi-airplane-fill text-success me-1" style="transform: rotate(-45deg);"></i>${r.dpt||''}</td>
            <td><i class="bi bi-airplane-fill text-danger me-1" style="transform: rotate(135deg);"></i>${r.arr||''}</td>
            <td>${r.dpt_time||''}</td>
            <td>${r.arr_time||''}</td>
            <td>${r.aircraft||''}</td>
            <td><a class="btn btn-sm btn-primary" href="/flights/${r.id}">Details</a> <button class=\"btn btn-sm btn-success ms-2 cc-bid\" data-flight-id=\"${r.id}\" data-dpt=\"${r.dpt||''}\" disabled>Bid</button></td>
          </tr>`).join('');
          document.getElementById('cc-schedules').style.display='';
          prepareBidButtons();
        });
      }

      // Click on a connecting destination: focus only that pair and reload schedules for dep->selected
      function onConnectingClick(dep, via, selected){
        setTopBox('cc-destination', selected);
        currentDest = selected;
        // Clear both connection candidates and the previous direct fan so we only show the chosen flight
        layerConnect.clearLayers();
        layerDirect.clearLayers();
        drawGeodesic(dep, selected, 'green', 4);
        try{ map.fitBounds(leaflet.featureGroup([leaflet.marker([dep.lat,dep.lon]), leaflet.marker([selected.lat,selected.lon])]).getBounds(), {padding:[40,40]}); }catch(e){}
        // reload schedules for the selected connection
        fetchJSON('/modules/destinations-map/schedules', { origin: dep.id, destination: selected.id, airline_id: airlineId }).then(rows=>{
          const tbody = document.querySelector('#cc-schedules-table tbody');
          tbody.innerHTML = rows.map(r=>`<tr>
            <td>${r.airline_logo?`<img src="${r.airline_logo}" alt="${r.airline||''}" style="height:16px;vertical-align:middle"/>`:''} <span class="ms-1">${r.ident||''}</span></td>
            <td>${r.airline||''}</td>
            <td><i class="bi bi-airplane-fill text-success me-1" style="transform: rotate(-45deg);"></i>${r.dpt||''}</td>
            <td><i class="bi bi-airplane-fill text-danger me-1" style="transform: rotate(135deg);"></i>${r.arr||''}</td>
            <td>${r.dpt_time||''}</td>
            <td>${r.arr_time||''}</td>
            <td>${r.aircraft||''}</td>
            <td><a class="btn btn-sm btn-primary" href="/flights/${r.id}">Details</a> <button class=\"btn btn-sm btn-success ms-2 cc-bid\" data-flight-id=\"${r.id}\" data-dpt=\"${r.dpt||''}\" disabled>Bid</button></td>
          </tr>`).join('');
          document.getElementById('cc-schedules').style.display='';
          prepareBidButtons();
        });
      }



      // ACARS overlay (same map; no second map instance)
      const aircraftIcon = leaflet.icon({ iconUrl: "{!! public_asset('/assets/img/acars/aircraft-aa.svg') !!}", iconSize:[42,42], iconAnchor:[21,21] });
      let acarsRouteLayer = null;
      function drawAcarsRoute(pirep_id){
        if(acarsRouteLayer){ map.removeLayer(acarsRouteLayer); acarsRouteLayer=null; }
        phpvms.request({ url: `/api/pireps/${pirep_id}/acars/geojson` }).then(resp=>{
          const route = resp.data.data;
          const g = new L.Geodesic([], { color:'#067ec1', weight:5, opacity:0.9, wrap:false }).addTo(map);
          g.fromGeoJson(route.line);
          acarsRouteLayer = g;
        });
      }
      function refreshAcars(){
        phpvms.request({ url:'/api/acars/geojson' }).then(resp=>{
          const flightGeoJson = resp.data.data;
          layerAcars.clearLayers();
          const gj = leaflet.geoJSON(flightGeoJson, {
            onEachFeature: (feature, layer) => {
              layer.on('click', ()=>{ drawAcarsRoute(feature.properties.pirep_id); showPirepInfo(feature.properties.pirep_id); });
              if (feature.properties && feature.properties.popup) { layer.bindPopup(feature.properties.popup); }
            },
            pointToLayer: (feature, latlng) => leaflet.marker(latlng, { icon: aircraftIcon, rotationAngle: feature.properties.heading })
          });
          layerAcars.addLayer(gj);
          phpvms.request({ url:'/api/acars' }).then(r=>{
            const n=(r.data.data||[]).length; document.getElementById('cc-flightcount').textContent= n? `${n} flight${n>1?'s':''} in progress` : 'No flights in progress';
            renderFlightsTable(r.data.data||[]);
          });
        });
      }
      function showPirepInfo(pirep_id){
        const box = document.getElementById('map-info-box');
        box.style.display='';
        phpvms.request({ url:`/api/pireps/${pirep_id}` }).then(resp=>{
          const p=resp.data.data;
          box.innerHTML = `<div class=\"border border-primary p-4 rounded\">
            <div class=\"d-flex align-items-center gap-3\">
              <img src=\"${(p.airline && p.airline.logo)||''}\" style=\"max-width:120px\"/>
              <div class=\"fw-bold\">${p.airline?.icao||''}${p.flight_number||''} — ${p.dpt_airport?.icao||''} to ${p.arr_airport?.icao||''}</div>
            </div>
          </div>`;
        });
      }
      function renderFlightsTable(pireps){
        const container = document.getElementById('live_flights');
        container.style.display='';
        const rows = (pireps||[]).map(p=>`<tr>
          <td style=\"width:46px\">${p.airline?.logo?`<img src='${p.airline.logo}' style='height:22px'/>`:''}</td>
          <td><a class=\"text-primary\" href=\"/users/${p.user?.id}\">${p.user?.name_private||''}</a></td>
          <td>${(p.airline?.icao||'')}${p.flight_number||''}</td>
          <td>${p.aircraft?.registration||''}</td>
          <td>${p.dpt_airport?.icao||''}</td>
          <td>${p.arr_airport?.icao||''}</td>
          <td>${(p.position?.altitude_msl||p.position?.altitude||0).toLocaleString()}</td>
          <td>${(p.position?.gs||0)}</td>
          <td>${(p.position && p.position.distance && (p.position.distance['{{ setting('units.distance') }}']||0))||0}</td>
          <td><span class=\"badge bg-success\">${p.status_text||''}</span></td>
        </tr>`).join('');
        container.innerHTML = `<div class=\"card card-flush mt-5 mb-5 border-hover-primary\"><div class=\"card-header\"><h3 class=\"card-title\">Flight Board</h3></div><div class=\"card-body p-0\"><div class=\"table-responsive\"><table class=\"table align-middle table-row-bordered table-row-solid gy-4 gs-9\"><thead class=\"border-gray-800 text-gray-800 fs-6 fw-semibold bg-lighten\"><tr><th></th><th>Pilot in Command</th><th>Flight</th><th>Aircraft</th><th>Departure</th><th>Arrival</th><th>Altitude</th><th>GS</th><th>Distance</th><th>Status</th></tr></thead><tbody>${rows}</tbody></table></div></div></div>`;
      }

      function setAcars(on){
        acarsOn = on;
        const info = document.getElementById('map-info-box');
        const board = document.getElementById('live_flights');
        document.getElementById('cc-flightcount').textContent='';
        if(on){ info.style.display=''; board.style.display=''; refreshAcars(); acarsTimer = setInterval(refreshAcars, {{ setting('acars.update_interval', 60) }} * 1000); }
        else { info.style.display='none'; board.style.display='none'; layerAcars.clearLayers(); if(acarsTimer){ clearInterval(acarsTimer); acarsTimer=null; } }
      }

      document.getElementById('cc-acars').addEventListener('click', ()=> setAcars(!acarsOn));
      document.getElementById('cc-reset').addEventListener('click', ()=>{ clearAll(); loadOrigins(); if(acarsOn) setAcars(false); });

      // init
      loadOrigins();
    })();
  </script>
@endsection

