<?php

namespace Modules\DestinationsMap\Http\Controllers\Api;

use App\Contracts\Controller;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Flight;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class MapController extends Controller
{
    public function airlines(Request $request): JsonResponse
    {
        $airlines = Airline::select('id','icao','name')->orderBy('icao')->get();
        return response()->json(['data' => $airlines]);
    }

    public function origins(Request $request): JsonResponse
    {
        $airlineId = $request->get('airline_id');
        $q = Flight::query()->with('dpt_airport')
            ->where('active', 1)->where('visible', 1);
        if ($airlineId) { $q->where('airline_id', $airlineId); }
        $rows = $q->select('dpt_airport_id')->distinct()->get();
        $airports = Airport::whereIn('id', $rows->pluck('dpt_airport_id')->all())
            ->select('id','icao','name','lat','lon')->orderBy('icao')->get();
        return response()->json(['data' => $airports]);
    }

    public function destinations(Request $request): JsonResponse
    {
        $origin = $request->get('origin');
        $airlineId = $request->get('airline_id');
        if (!$origin) { return response()->json(['data' => []]); }
        $q = Flight::query()->with('arr_airport')
            ->where('dpt_airport_id', $origin)
            ->where('active', 1)->where('visible', 1);
        if ($airlineId) { $q->where('airline_id', $airlineId); }
        $rows = $q->select('arr_airport_id')->distinct()->get();
        $airports = Airport::whereIn('id', $rows->pluck('arr_airport_id')->all())
            ->select('id','icao','name','lat','lon')->orderBy('icao')->get();
        return response()->json(['data' => $airports]);
    }

    public function connecting(Request $request): JsonResponse
    {
        $airport = $request->get('airport');
        $exclude = $request->get('exclude'); // origin to exclude
        $airlineId = $request->get('airline_id');
        if (!$airport) { return response()->json(['data' => []]); }
        $q = Flight::query()->with('arr_airport')
            ->where('dpt_airport_id', $airport)
            ->where('active', 1)->where('visible', 1);
        if ($exclude) { $q->where('arr_airport_id', '!=', $exclude); }
        if ($airlineId) { $q->where('airline_id', $airlineId); }
        $rows = $q->select('arr_airport_id')->distinct()->get();
        $airports = Airport::whereIn('id', $rows->pluck('arr_airport_id')->all())
            ->select('id','icao','name','lat','lon')->orderBy('icao')->get();
        return response()->json(['data' => $airports]);
    }

    public function schedules(Request $request): JsonResponse
    {
        $origin = $request->get('origin');
        $destination = $request->get('destination');
        $airlineId = $request->get('airline_id');
        if (!$origin || !$destination) { return response()->json(['data' => []]); }

        $q = Flight::query()->with(['airline','dpt_airport','arr_airport','subfleets'])
            ->where('dpt_airport_id', $origin)
            ->where('arr_airport_id', $destination)
            ->where('active', 1)->where('visible', 1);
        if ($airlineId) { $q->where('airline_id', $airlineId); }

        $flights = $q->orderBy('flight_number')->limit(200)->get()->map(function($f){
            return [
                'id' => $f->id,
                'ident' => $f->airline?->icao.$f->flight_number,
                'airline' => $f->airline?->icao,
                'airline_logo' => $f->airline?->logo,
                'dpt' => $f->dpt_airport?->icao,
                'arr' => $f->arr_airport?->icao,
                'dpt_time' => $f->dpt_time,
                'arr_time' => $f->arr_time,
                'route' => $f->route,
                'aircraft' => $f->subfleets?->pluck('type')->filter()->values()->implode(', '),
            ];
        });

        return response()->json(['data' => $flights]);
    }
}

