<?php

namespace Modules\ModuleCenter\Http\Controllers\Admin;

use App\Contracts\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Str;
use Modules\ModuleCenter\Services\CatalogService;
use Modules\ModuleCenter\Services\InstallerService;
use Modules\ModuleCenter\Services\LicenseService;
use Modules\ModuleCenter\Services\PreflightService;

class ModuleCenterController extends Controller
{
    public function index(Request $request)
    {
        // Installed view
        $preflight = app(PreflightService::class)->run();
        $installed = $this->scanInstalled();
        $catalog   = app(CatalogService::class)->getCatalog();

        // Build normalized catalog index (supports alias with or without version suffix)
        $catalog_index = [];
        foreach ($catalog as $it) {
            $aliasRaw = $it['alias'] ?? ($it['name'] ?? '');
            $keyMain  = $this->normalizeKey($aliasRaw);
            if ($keyMain) { $catalog_index[$keyMain] = $it; }
            // Also map alias without trailing version like "pbs-1.0" => "pbs"
            $aliasNoVer = preg_replace('/-[0-9]+(\.[0-9]+)*$/', '', strtolower((string)$aliasRaw));
            $keyNoVer   = $this->normalizeKey($aliasNoVer);
            if ($keyNoVer && $keyNoVer !== $keyMain) { $catalog_index[$keyNoVer] = $it; }
        }

        // Only show installed modules that exist in our catalog
        $filtered = array_values(array_filter($installed, function ($m) use ($catalog_index) {
            $lookup = $m['lookup'] ?? '';
            return $lookup !== '' && isset($catalog_index[$lookup]);
        }));
        $installed = $filtered;

        // Enrich installed with catalog info
        $licenses = app(LicenseService::class)->all();
        foreach ($installed as &$m) {
            $key = $m['lookup'];
            $cat = $catalog_index[$key] ?? null;
            $m['installed_ok']    = true; // presence of module.json
            $m['latest_version']  = $cat['version'] ?? null;
            $m['description']     = $cat['description'] ?? null;
            $m['update_available']= $cat && $m['version'] && $cat['version']
                                   ? version_compare((string)$cat['version'], (string)$m['version'], '>')
                                   : false;
            $m['paid']            = (bool)($cat['paid'] ?? false);
            $aliasForLic          = $cat['alias'] ?? $m['alias'];
            $m['unlocked']        = !$m['paid'] || isset($licenses[$aliasForLic] );
            $m['can_update']      = $m['update_available'] && $m['unlocked'];
        }
        unset($m);

        return view('modulecenter::admin.installed', [
            'preflight'      => $preflight,
            'installed'      => $installed,
            'catalog_index'  => $catalog_index,
            'needs_update'   => session('modulecenter_needs_update', false),
        ]);
    }

    public function catalog(Request $request)
    {
        $preflight = app(PreflightService::class)->run();
        $catalog   = app(CatalogService::class)->getCatalog();
        $licenses  = app(LicenseService::class)->all();
        $installed = $this->scanInstalled();

        // Build quick lookup by normalized alias/slug
        $installedIndex = [];
        foreach ($installed as $m) {
            $installedIndex[$this->normalizeKey($m['alias'] ?: $m['name'])] = $m;
        }

        foreach ($catalog as &$item) {
            $alias = $item['alias'] ?? ($item['slug'] ?? null);
            $name  = $item['name']  ?? $alias;
            $norm  = $this->normalizeKey($alias ?: $name ?: '');
            $match = $norm && isset($installedIndex[$norm]) ? $installedIndex[$norm] : null;
            $item['installed'] = (bool)$match;
            $item['installed_version'] = $match['version'] ?? null;
            $item['update_available'] = $match && !empty($item['version']) && !empty($match['version']) && version_compare((string)$item['version'], (string)$match['version'], '>');
            $item['unlocked'] = !$this->isPaid($item) || isset($licenses[$alias ?? '']);
        }

        return view('modulecenter::admin.catalog', [
            'preflight' => $preflight,
            'catalog'   => $catalog,
            'licenses'  => $licenses,
            'needs_update' => session('modulecenter_needs_update', false),
        ]);
    }

    private function scanInstalled(): array
    {
        $modulesPath = base_path('modules');
        $out = [];
        if (!is_dir($modulesPath)) { return $out; }
        foreach (scandir($modulesPath) as $dir) {
            if ($dir === '.' || $dir === '..') { continue; }
            $manifestPath = $modulesPath.'/'.$dir.'/module.json';
            if (!file_exists($manifestPath)) { continue; }
            try {
                $manifest = json_decode((string)file_get_contents($manifestPath), true);
            } catch (\Throwable $e) { $manifest = null; }
            if (!is_array($manifest)) { continue; }
            $name  = $manifest['name'] ?? $dir;
            $alias = $manifest['alias'] ?? ($manifest['slug'] ?? Str::slug($name));

            // Derive version: prefer module.json, then manifest.json fallback
            $version = $manifest['version'] ?? ($manifest['Version'] ?? ($manifest['ver'] ?? ($manifest['module_version'] ?? null)));
            // Tolerate BOM/zero-width chars or odd punctuation around the key name
            if ($version === null) {
                foreach ($manifest as $k => $v) {
                    $clean = strtolower(preg_replace('/[^a-z0-9_]/', '', (string)$k));
                    if ($clean === 'version') { $version = is_scalar($v) ? (string)$v : null; break; }
                }
            }
            if ($version === null) {
                $mf = $modulesPath.'/'.$dir.'/manifest.json';
                if (file_exists($mf)) {
                    try {
                        $m2 = json_decode((string)file_get_contents($mf), true);
                        if (is_array($m2)) { $version = $m2['version'] ?? ($m2['Version'] ?? null); }
                    } catch (\Throwable $e) { /* ignore */ }
                }
            }
            // Last-resort: parse raw module.json text for a "version" pattern if JSON keys are mangled
            if ($version === null) {
                try {
                    $raw = (string)file_get_contents($manifestPath);
                    if (preg_match('/\"\s*version\s*\"\s*:\s*\"([^\"]+)\"/i', $raw, $m)) {
                        $version = $m[1];
                    } elseif (preg_match('/version\s*[:=]\s*\"?([0-9][^\"\s,}]*)/i', $raw, $m)) {
                        $version = $m[1];
                    }
                } catch (\Throwable $e) { /* ignore */ }
            }

            // Determine active status: prefer phpVMS ModuleService (DB + module system),
            // fallback to manifest flags, default true so the cell is never blank
            try {
                $moduleSvc = app('App\\Services\\ModuleService');
                $isActiveDb = method_exists($moduleSvc, 'isModuleActive') ? (bool)$moduleSvc->isModuleActive($name) : null;
            } catch (\Throwable $e) { $isActiveDb = null; }
            $active = $isActiveDb ?? (isset($manifest['active']) ? (bool)$manifest['active'] : (isset($manifest['enabled']) ? (bool)$manifest['enabled'] : false));

            $out[] = [
                'folder'  => $dir,
                'name'    => $name,
                'alias'   => $alias,
                'version' => $version,
                'active'  => $active,
                'lookup'  => $this->normalizeKey($alias ?: $name),
            ];
        }
        // Exclude Module Center itself from listing
        $out = array_values(array_filter($out, function ($m) {
            return $this->normalizeKey($m['alias']) !== 'modulecenter' && $this->normalizeKey($m['name']) !== 'modulecenter';
        }));
        // Sort by name
        usort($out, fn($a,$b) => strcasecmp($a['name'], $b['name']));
        return $out;
    }


    private function normalizeKey(?string $s): string
    {
        $s = (string)$s;
        $s = strtolower($s);
        return preg_replace('/[^a-z0-9]/', '', $s) ?? '';
    }

    public function refreshCatalog(Request $request)
    {
        $remote = app(CatalogService::class)->refresh();
        $msg = $remote ? 'Catalog refreshed from remote' : 'Catalog refreshed (remote unavailable or invalid; using cached/built-in)';
        return Redirect::back()->with('status', $msg);
    }

    public function unlock(Request $request)
    {
        $request->validate([
            'alias' => 'required|string',
	            'key'   => 'required|string',
        ]);
        $alias = trim((string)$request->input('alias'));
        $key   = trim((string)$request->input('key'));

        $lic = app(LicenseService::class);
	        // Verify the provided key against the remote API (domain-bound)
	        if (!$lic->verify($alias, $key)) {
            return Redirect::back()->withErrors(['key' => 'Invalid license key']);
        }
	        if (!$lic->store($alias, $key)) {
            return Redirect::back()->withErrors(['key' => 'Unable to save key']);
        }
        return Redirect::back()->with('status', 'License key verified and saved');
    }

    public function install(Request $request)
    {
        $request->validate(['alias' => 'required|string']);
        $alias = $request->input('alias');
        $catalog = app(CatalogService::class)->getCatalogByAlias($alias);
        if (!$catalog) {
            return Redirect::back()->withErrors(['alias' => 'Module not found in catalog']);
        }

        $installer = app(InstallerService::class);
        if ($this->isPaid($catalog)) {
            $licenses = app(LicenseService::class)->all();
            if (!isset($licenses[$alias])) {
                return Redirect::back()->withErrors(['alias' => 'This module requires unlocking before install']);
            }
            // Paid modules are downloaded via API using alias+key
            $result = $installer->installPaid($alias, $licenses[$alias]);
        } else {
            $zipUrl = $catalog['zip_url'] ?? null;
            if (!$zipUrl) {
                return Redirect::back()->withErrors(['alias' => 'No ZIP URL defined for this module']);
            }
            $result = $installer->installFromUrl($zipUrl);
        }
        if (!$result['ok']) {
            return Redirect::back()->withErrors(['install' => $result['error'] ?? 'Install failed']);
        }

        session(['modulecenter_needs_update' => true]);
        return Redirect::back()->with('status', 'The module was installed successfully, please head to the Modules section to activate the new module and then run phpvms/update to run the migrations');

    }

    public function installFromUrl(Request $request)
    {
        $request->validate(['url' => 'required|url']);
        $res = app(InstallerService::class)->installFromUrl($request->input('url'));
        if (!$res['ok']) {
            return Redirect::back()->withErrors(['install' => $res['error'] ?? 'Install failed']);
        }
        session(['modulecenter_needs_update' => true]);
        return Redirect::back()->with('status', 'Module installed from URL. Please click "Run updates" to complete.');
    }

    public function download(Request $request)
    {
        $request->validate(['alias' => 'required|string']);
        $alias = $request->input('alias');
        $cat   = app(CatalogService::class)->getCatalogByAlias($alias);
        if (!$cat) {
            return Redirect::back()->withErrors(['download' => 'Module not found in catalog']);
        }

        // Free modules: redirect directly to the ZIP
        if (!(bool)($cat['paid'] ?? false)) {
            $zip = (string)($cat['zip_url'] ?? '');
            if ($zip === '') {
                return Redirect::back()->withErrors(['download' => 'No ZIP URL available for this module']);
            }
            return Redirect::away($zip);
        }

        // Paid modules: require license and construct API URL with key
        $licenses = app(LicenseService::class)->all();
        $key = $licenses[$alias] ?? null;
        if (!$key) {
            return Redirect::back()->withErrors(['download' => 'This module requires unlocking before download']);
        }
	        // Include current domain so the shop can enforce domain-bound
	        // licensing when serving the ZIP.
	        $domain = '';
	        try {
	            $licSvc = app(LicenseService::class);
	            if (method_exists($licSvc, 'getCurrentDomain')) {
	                $domain = (string) $licSvc->getCurrentDomain();
	            }
	        } catch (\Throwable $e) {
	            $domain = '';
	        }
	        $params = ['alias' => $alias, 'slug' => $alias, 'key' => $key];
	        if ($domain !== '') {
	            $params['domain'] = $domain;
	        }
	        $qs = http_build_query($params);
	        $url = 'https://simaddon.space/modulecenter/api/download.php?'.$qs;
        return Redirect::away($url);
    }

    private function isPaid(array $item): bool
    {
        return (bool)($item['paid'] ?? false);
    }
}

