<?php

namespace Modules\ModuleCenter\Services;

use Illuminate\Support\Facades\Log;

class CatalogService
{
    private const REMOTE_CATALOG_URL = 'https://simaddon.space/modulecenter/catalog/catalog.json';

    private string $cachePath;
    private string $builtInPath;

    public function __construct()
    {
        $this->cachePath   = storage_path('app/modulecenter/catalog.json');
        $this->builtInPath = __DIR__.'/../Resources/catalog.json';
    }

    public function getCatalog(): array
    {
        // Try remote first; fall back to cache then built-in
        $remote = $this->fetchRemote();
        if ($remote) {
            // Use ONLY the remote data; do not backfill from built-in to avoid masking catalog issues
            $normalized = $this->normalizeCatalog($remote);
            $this->writeCache($normalized);
            return $normalized;
        }

        $data = $this->readJson($this->cachePath);
        if (!$data) {
            $data = $this->readJson($this->builtInPath);
        }
        // If remote not available, fall back to cache or built-in as-is (no field backfill)
        return $this->normalizeCatalog(is_array($data) ? $data : []);
    }

    /**
     * Fill missing fields (e.g., description) from the built-in catalog as a fallback,
     * without overriding values already present from remote.
     */
    private function withBuiltInFallback(array $list): array
    {
        $builtIn = $this->readJson($this->builtInPath) ?: [];
        $fallback = $this->normalizeCatalog(is_array($builtIn) ? $builtIn : []);
        if (empty($fallback)) { return $list; }

        // Build index by normalized alias and alias without trailing version suffix
        $fbIndex = [];
        foreach ($fallback as $it) {
            $aliasRaw = $it['alias'] ?? ($it['name'] ?? '');
            $keyMain  = $this->normalizeKey($aliasRaw);
            if ($keyMain) { $fbIndex[$keyMain] = $it; }
            $aliasNoVer = preg_replace('/-[0-9]+(\.[0-9]+)*$/', '', strtolower((string)$aliasRaw));
            $keyNoVer   = $this->normalizeKey($aliasNoVer);
            if ($keyNoVer && $keyNoVer !== $keyMain) { $fbIndex[$keyNoVer] = $it; }
        }

        foreach ($list as &$it) {
            $aliasRaw = $it['alias'] ?? ($it['name'] ?? '');
            $key = $this->normalizeKey($aliasRaw);
            $fb  = $fbIndex[$key] ?? null;
            if (!$fb && $aliasRaw) {
                $aliasNoVer = preg_replace('/-[0-9]+(\.[0-9]+)*$/', '', strtolower((string)$aliasRaw));
                $fb = $fbIndex[$this->normalizeKey($aliasNoVer)] ?? null;
            }
            if ($fb) {
                if (empty($it['description']) && !empty($fb['description'])) {
                    $it['description'] = $fb['description'];
                }
            }
        }
        unset($it);
        return $list;
    }

    private function normalizeKey(?string $s): string
    {
        $s = (string)$s;
        $s = strtolower($s);
        return preg_replace('/[^a-z0-9]/', '', $s) ?? '';
    }

    public function getCatalogByAlias(string $alias): ?array
    {
        foreach ($this->getCatalog() as $item) {
            $itemAlias = $item['alias'] ?? ($item['slug'] ?? null);
            if ($itemAlias === $alias) {
                return $item;
            }
        }
        return null;
    }

    /**
     * Normalize catalog JSON from different sources to a flat array of items
     * with keys: name, alias, paid, version, zip_url, description (optional)
     */
    private function normalizeCatalog(array $data): array
    {
        // If the remote returns {"modules": [...]}, extract it
        if (array_key_exists('modules', $data) && is_array($data['modules'])) {
            $data = $data['modules'];
        }
        if (!is_array($data)) { return []; }
        $out = [];
        foreach ($data as $item) {
            if (!is_array($item)) { continue; }
            $alias = $item['alias'] ?? ($item['slug'] ?? null);
            $name  = $item['name']  ?? ($alias ?: null);
            if (!$name) { continue; }
            $out[] = [
                'name'        => $name,
                'alias'       => $alias ?? $name,
                'paid'        => (bool)($item['paid'] ?? false),
                'version'     => $item['version'] ?? null,
                'zip_url'     => $item['zip_url'] ?? ($item['download_url'] ?? null),
                'description' => $item['description'] ?? ($item['desc'] ?? null),
            ];
        }
        return $out;
    }


    public function refresh(): bool
    {
        // Remove local cache first so we don't accidentally read stale data later
        if (file_exists($this->cachePath)) {
            @unlink($this->cachePath);
        }

        $remote = $this->fetchRemote();
        if ($remote) {
            $this->writeCache($this->normalizeCatalog($remote));
            return true; // refreshed from remote
        }
        // fallback to built-in when remote unavailable
        if (!is_dir(dirname($this->cachePath))) {
            @mkdir(dirname($this->cachePath), 0775, true);
        }
        $data = $this->readJson($this->builtInPath) ?: [];
        file_put_contents($this->cachePath, json_encode($data, JSON_PRETTY_PRINT));
        return false; // used built-in
    }

    private function fetchRemote(): ?array
    {
        try {
            // Strong cache-bust to avoid CDN/proxy stale responses
            $url = self::REMOTE_CATALOG_URL;
            $url .= (str_contains($url, '?') ? '&' : '?').('no_cache='.rawurlencode(uniqid('', true)).'&t='.microtime(true));
            $ctx = stream_context_create([
                'http' => [
                    'timeout' => 10,
                    'header'  => "Cache-Control: no-cache, no-store, must-revalidate\r\nPragma: no-cache\r\nExpires: 0\r\nUser-Agent: ModuleCenter/1.0\r\n",
                ],
            ]);
            $raw = @file_get_contents($url, false, $ctx);
            if ($raw === false) { return null; }
            $json = json_decode($raw, true);
            if (!is_array($json)) {
                Log::warning('ModuleCenter remote catalog JSON invalid. Sample: '.substr((string)$raw, 0, 200));
                return null;
            }
            return $json;
        } catch (\Throwable $e) {
            Log::info('ModuleCenter remote catalog unavailable: '.$e->getMessage());
            return null;
        }
    }

    private function writeCache(array $data): void
    {
        if (!is_dir(dirname($this->cachePath))) {
            @mkdir(dirname($this->cachePath), 0775, true);
        }
        @file_put_contents($this->cachePath, json_encode($data, JSON_PRETTY_PRINT));
    }

    private function readJson(string $path): ?array
    {
        if (!file_exists($path)) {
            return null;
        }
        try {
            $raw = file_get_contents($path);
            $json = json_decode($raw, true);
            return is_array($json) ? $json : null;
        } catch (\Throwable $e) {
            Log::warning('ModuleCenter catalog read failed: '.$e->getMessage());
            return null;
        }
    }
}

