<?php

namespace Modules\ModuleCenter\Services;

class LicenseService
{
    private const VERIFY_URL = 'https://simaddon.space/modulecenter/api/license.php';
    private const REMOTE_LICENSES_URL = 'https://simaddon.space/modulecenter/storage/licenses.json';

    private string $path;

    public function __construct()
    {
        $this->path = storage_path('app/modulecenter/licenses.json');
    }

    /**
     * Return saved licenses as alias => key map. Supports both map format
     * and {"licenses": [{"slug"|"alias", "key", "status"}]} for convenience.
     */
    public function all(): array
    {
        $paths = $this->candidateLicensePaths();
        foreach ($paths as $p) {
            if (!file_exists($p)) { continue; }
            $raw = @file_get_contents($p);
            $json = json_decode((string)$raw, true);
            if (!is_array($json)) { continue; }
            // Map format
            if (isset($json['licenses']) && is_array($json['licenses'])) {
                $map = [];
                foreach ($json['licenses'] as $row) {
                    if (!is_array($row)) { continue; }
                    $alias = (string)($row['slug'] ?? ($row['alias'] ?? ''));
                    $key   = (string)($row['key'] ?? '');
                    $status= strtolower((string)($row['status'] ?? 'active'));
                    if ($alias !== '' && $key !== '' && $status !== 'revoked' && $status !== 'disabled') {
                        $map[$alias] = $key;
                    }
                }
                if (!empty($map)) { return $map; }
            }
            // Plain map { alias: key }
            $isAssoc = !empty($json) && array_keys($json) !== range(0, count($json)-1);
            if ($isAssoc) {
                return $json;
            }
        }
        return [];
    }

    public function store(string $alias, string $key): bool
    {
        $all = $this->all();
        $all[$alias] = $key;
        if (!is_dir(dirname($this->path))) {
            @mkdir(dirname($this->path), 0775, true);
        }
        return (bool)file_put_contents($this->path, json_encode($all, JSON_PRETTY_PRINT));
    }

    /**
     * Verify a license:
     * - First try remote verification with the provided key
	     * - If it fails, optionally fall back to the remote licenses list while still
	     *   enforcing domain-bound checks.
     */
    public function verify(string $alias, string $key): bool
    {
	        $key = trim((string)$key);
	        if ($key === '') {
	            return false;
	        }
	        // Primary: remote API, which enforces slug+key+domain
	        if ($this->verifyRemote($alias, $key)) {
	            return true;
	        }
	        // Fallback: remote licenses list, with the same domain constraint
	        return $this->verifyAgainstRemoteList($alias, $key);
    }

	    /**
	     * Resolve and return the key from explicit user input.
	     *
	     * For security and domain binding, we no longer auto-resolve keys from the
	     * remote list based only on alias. The user must enter the exact key that is
	     * active for this VA/domain.
	     */
    public function resolveKey(string $alias, ?string $inputKey): ?string
    {
	        $inputKey = trim((string)$inputKey);
	        if ($inputKey === '') {
	            return null;
	        }
	        return $this->verifyRemote($alias, $inputKey) ? $inputKey : null;
    }

    private function verifyRemote(string $alias, string $key): bool
    {
        try {
	            $payload = [
	                'alias'  => $alias,
	                'slug'   => $alias,
	                'key'    => $key,
	                // Include current domain so server can enforce domain-bound licenses
	                'domain' => $this->currentDomain(),
	            ];
	            $ctx = stream_context_create([
	                'http' => [
	                    'timeout' => 10,
	                    'method'  => 'POST',
	                    'header'  => "Content-Type: application/json\r\nUser-Agent: ModuleCenter/1.0\r\n",
	                    'content' => json_encode($payload),
	                ],
	            ]);
	            $raw = @file_get_contents(self::VERIFY_URL, false, $ctx);
	            if ($raw === false) { return false; }
	            $rawTrim = trim($raw);
	            $json = json_decode($rawTrim, true);
	            if (is_array($json)) {
	                if (isset($json['valid']))  { return (bool)$json['valid']; }
	                if (isset($json['ok']))     { return (bool)$json['ok']; }
	                if (isset($json['status'])) { return strtolower((string)$json['status']) === 'ok'; }
	                if (isset($json['message'])){ return strcasecmp((string)$json['message'], 'valid') === 0; }
	            }
	            return strcasecmp($rawTrim, 'OK') === 0 || strcasecmp($rawTrim, 'VALID') === 0;
        } catch (\Throwable $e) {
            return false;
        }
    }

	    /**
	     * Derive the current VA domain for domain-bound license checks.
	     * Falls back to APP_URL host, then current HTTP host.
	     */
	    private function currentDomain(): string
	    {
	        try {
	            // Prefer configured app.url
	            $url = (string) (config('app.url') ?? '');
	            if ($url !== '') {
	                if (!preg_match('~^https?://~i', $url)) {
	                    $url = 'https://'.$url;
	                }
	                $parts = @parse_url($url);
	                $host  = $parts['host'] ?? '';
	                if ($host !== '') { return strtolower($host); }
	            }
	        } catch (\Throwable $e) {
	            // ignore and fallback
	        }
	        $host = isset($_SERVER['HTTP_HOST']) ? (string)$_SERVER['HTTP_HOST'] : '';
	        $host = strtolower(trim($host));
	        $host = preg_replace('/:.*/', '', $host); // strip port
	        if ($host !== '') { return $host; }
	        return '';
	    }

	    /**
	     * Public accessor for the normalized current domain so other services
	     * (e.g. InstallerService, controllers) can reuse the same logic that
	     * the license verification flow uses.
	     */
	    public function getCurrentDomain(): string
	    {
	        return $this->currentDomain();
	    }

	    private function resolveKeyFromRemote(string $alias): ?string
    {
        try {
            $ctx = stream_context_create(['http' => ['timeout' => 15, 'header' => "User-Agent: ModuleCenter/1.0\r\n"]]);
            $url1 = self::REMOTE_LICENSES_URL;
            $url2 = str_replace('/licenses.json', '/license.json', $url1);
            $raw = @file_get_contents($url1, false, $ctx);
            if ($raw === false) { $raw = @file_get_contents($url2, false, $ctx); }
            if ($raw === false) { return null; }
	            $json = json_decode(trim((string)$raw), true);
            if (!is_array($json)) { return null; }
	
	            // Build candidate aliases: with and without trailing version suffix
            $candidates = [$alias];
            $noVer = preg_replace('/-[0-9]+(\.[0-9]+)*$/', '', (string)$alias);
            if ($noVer !== '' && $noVer !== $alias) { $candidates[] = $noVer; }
	            $domain = strtolower($this->currentDomain());

	            // Accept {"licenses":[...]} or direct map
            if (isset($json['licenses']) && is_array($json['licenses'])) {
                foreach ($json['licenses'] as $row) {
                    if (!is_array($row)) { continue; }
                    $slug   = (string)($row['slug'] ?? ($row['alias'] ?? ''));
                    $key    = (string)($row['key'] ?? '');
                    $status = strtolower((string)($row['status'] ?? 'active'));
	                    $rDomain = strtolower(trim((string)($row['domain'] ?? '')));
	                    if ($rDomain !== '') {
	                        // Stored domain must match current domain; if we don't
	                        // know our domain, treat this entry as not applicable.
	                        if ($domain === '' || $rDomain !== $domain) { continue; }
	                    }
	                    if (in_array($slug, $candidates, true) && $key !== '' && $status !== 'revoked' && $status !== 'disabled') {
	                        return $key;
                    }
                }
            } else {
                // Map format { alias: key }
                foreach ($candidates as $cand) {
                    if (isset($json[$cand]) && is_string($json[$cand]) && $json[$cand] !== '') {
                        return (string)$json[$cand];
                    }
                }
            }
        } catch (\Throwable $e) {
            return null;
        }
        return null;
    }

	    private function verifyAgainstRemoteList(string $alias, string $key): bool
    {
        try {
            $ctx = stream_context_create(['http' => ['timeout' => 15, 'header' => "User-Agent: ModuleCenter/1.0\r\n"]]);
            $url1 = self::REMOTE_LICENSES_URL;
            $url2 = str_replace('/licenses.json', '/license.json', $url1);
            $raw = @file_get_contents($url1, false, $ctx);
            if ($raw === false) { $raw = @file_get_contents($url2, false, $ctx); }
            if ($raw === false) { return false; }
	            $json = json_decode(trim((string)$raw), true);
            if (!is_array($json)) { return false; }

            $candidates = [$alias];
            $noVer = preg_replace('/-[0-9]+(\.[0-9]+)*$/', '', (string)$alias);
            if ($noVer !== '' && $noVer !== $alias) { $candidates[] = $noVer; }
            $normKey = preg_replace('/[\x{2010}-\x{2015}\x{2212}\x{FE63}\x{FF0D}]/u', '-', trim((string)$key));
	            $domain  = strtolower($this->currentDomain());

	            if (isset($json['licenses']) && is_array($json['licenses'])) {
	                foreach ($json['licenses'] as $row) {
	                    if (!is_array($row)) { continue; }
	                    $slug    = (string)($row['slug'] ?? ($row['alias'] ?? ''));
	                    $k       = preg_replace('/[\x{2010}-\x{2015}\x{2212}\x{FE63}\x{FF0D}]/u', '-', trim((string)($row['key'] ?? '')));
	                    $status  = strtolower((string)($row['status'] ?? 'active'));
	                    $rDomain = strtolower(trim((string)($row['domain'] ?? '')));
	                    if ($rDomain !== '') {
	                        if ($domain === '' || $rDomain !== $domain) { continue; }
	                    }
	                    if (in_array($slug, $candidates, true) && $k !== '' && hash_equals($k, $normKey) && $status !== 'revoked' && $status !== 'disabled') {
	                        return true;
	                    }
	                }
            } else {
                foreach ($candidates as $cand) {
                    if (isset($json[$cand]) && is_string($json[$cand])) {
                        $k = preg_replace('/[\x{2010}-\x{2015}\x{2212}\x{FE63}\x{FF0D}]/u', '-', trim((string)$json[$cand]));
                        if ($k !== '' && hash_equals($k, $normKey)) { return true; }
                    }
                }
            }
        } catch (\Throwable $e) {
            return false;
        }
        return false;
    }

    public function has(string $alias): bool
    {
        $all = $this->all();
        return array_key_exists($alias, $all);
    }

    private function candidateLicensePaths(): array
    {
        return [
            // Stored map used by unlock flow
            $this->path,
            // Support dev/test whitelist formats and locations
            storage_path('app/modulecenter/license.json'),
            base_path('Modules/ModuleCenter/Resources/license.json'),
            base_path('modules/ModuleCenter/Resources/license.json'),
            base_path('Modules/ModuleCenter/Resources/licenses.json'),
            base_path('modules/ModuleCenter/Resources/licenses.json'),
            base_path('modules/modulecenter/Resources/license.json'),
        ];
    }

    private function verifyLocal(string $alias, string $key): bool
    {
        $alias = (string)$alias;
        $key   = (string)$key;
        if ($alias === '' || $key === '') { return false; }
        foreach ($this->candidateLicensePaths() as $p) {
            if (!file_exists($p)) { continue; }
            $raw = @file_get_contents($p);
            $json = json_decode((string)$raw, true);
            if (!is_array($json)) { continue; }
            // Map format
            if (isset($json['licenses']) && is_array($json['licenses'])) {
                foreach ($json['licenses'] as $row) {
                    if (!is_array($row)) { continue; }
                    $slug   = (string)($row['slug'] ?? ($row['alias'] ?? ''));
                    $k      = (string)($row['key'] ?? '');
                    $status = strtolower((string)($row['status'] ?? 'active'));
                    if ($slug === $alias && $k === $key && $status !== 'revoked' && $status !== 'disabled') {
                        return true;
                    }
                }
            }
            // Plain map { alias: key }
            $isAssoc = !empty($json) && array_keys($json) !== range(0, count($json)-1);
            if ($isAssoc && isset($json[$alias]) && (string)$json[$alias] === $key) {
                return true;
            }
        }
        return false;
    }
}

