<?php

namespace Modules\ModuleCenter\Services;

use Illuminate\Support\Facades\Log;
use ZipArchive;

class InstallerService
{
    private const PAID_DOWNLOAD_URL = 'https://simaddon.space/modulecenter/api/download.php';

    public function installFromUrl(string $url): array
    {
        try {
            $dlDir = storage_path('app/module_downloads');
            if (!is_dir($dlDir)) { @mkdir($dlDir, 0775, true); }
            $tmp = $dlDir.'/module_'.date('Ymd_His').'.zip';

            // Prefer PHP streams to avoid requiring Guzzle; add strong cache-busting
            $url2 = $url.(str_contains($url, '?') ? '&' : '?').'no_cache='.rawurlencode(uniqid('', true)).'&t='.microtime(true);
            $ctx = stream_context_create([
                'http' => [
                    'timeout' => 60,
                    'header'  => "Cache-Control: no-cache, no-store, must-revalidate\r\nPragma: no-cache\r\nExpires: 0\r\nUser-Agent: ModuleCenter/1.0\r\n",
                ],
            ]);
            $data = @file_get_contents($url2, false, $ctx);
            if ($data === false) {
                return ['ok' => false, 'error' => 'Download failed'];
            }
            file_put_contents($tmp, $data);
            return $this->installFromZip($tmp);
        } catch (\Throwable $e) {
            Log::error('Module installFromUrl failed: '.$e->getMessage());
            return ['ok' => false, 'error' => 'Install error'];
        }
    }

    public function installPaid(string $alias, string $key): array
    {
        try {
	            // Include current VA domain so the remote download endpoint can
	            // enforce domain-bound licenses in the same way as the
	            // verification API.
	            $domain = '';
	            try {
	                $licSvc = app(\Modules\ModuleCenter\Services\LicenseService::class);
	                if (method_exists($licSvc, 'getCurrentDomain')) {
	                    $domain = (string) $licSvc->getCurrentDomain();
	                }
	            } catch (\Throwable $e) {
	                $domain = '';
	            }
	            $params = ['alias' => $alias, 'slug' => $alias, 'key' => $key];
	            if ($domain !== '') {
	                $params['domain'] = $domain;
	            }
	            $qs = http_build_query($params);
	            $url = self::PAID_DOWNLOAD_URL.'?'.$qs;
            return $this->installFromUrl($url);
        } catch (\Throwable $e) {
            Log::error('Module installPaid failed: '.$e->getMessage());
            return ['ok' => false, 'error' => 'Paid download failed'];
        }
    }

    public function installFromZip(string $zipPath): array
    {
        try {
            $debug = (bool) env('MODULECENTER_DEBUG', false);

            $zip = new ZipArchive();
        if ($zip->open($zipPath) !== true) {
            return ['ok' => false, 'error' => 'Unable to open ZIP'];
        }

        // Find module.json (very tolerant), fallback to manifest.json; log a small sample for debugging

            // BYPASS: If ZIP has exactly one top-level folder, copy it 1:1 and stop
            $topDirs = [];
            $hasRootFiles = false;
            for ($i = 0; $i < $zip->numFiles; $i++) {
                $n = (string)$zip->getNameIndex($i);
                if ($n === '') { continue; }
                $p = str_replace(['\\'], '/', $n);
                $p = ltrim($p, '/');
                if ($p === '' || $p === '.' || $p === '..') { continue; }
                if (strpos($p, '__MACOSX/') === 0) { continue; }
                $pos = strpos($p, '/');
                if ($pos === false) {
                    if (substr($p, -1) !== '/') { $hasRootFiles = true; }
                    continue;
                }
                $top = substr($p, 0, $pos);
                if ($top !== '' && $top !== '.' && $top !== '..' && $top !== '__MACOSX') {
                    $topDirs[$top] = true;
                }
            }
            if (!$hasRootFiles && count($topDirs) === 1) {
                $folderName = trim(basename(array_key_first($topDirs)));
                if ($folderName === '' || $folderName === '.' || $folderName === '..' || strpos($folderName, '/') !== false || strpos($folderName, '\\') !== false) {
                    $zip->close();
                    return ['ok' => false, 'error' => 'Invalid module folder in ZIP'];
                }
                $tmpBase = storage_path('app/module_tmp');
                if (!is_dir($tmpBase)) { @mkdir($tmpBase, 0775, true); }
                $tmpDir = $tmpBase.'/zip_'.date('Ymd_His').'_'.uniqid();
                @mkdir($tmpDir, 0775, true);
                // Verify temp dir writable (prevents silent extract failures)
                if (!@is_writable($tmpDir)) {
                    $zip->close();
                    \Log::error('ModuleCenter: temp not writable', ['tmpDir' => $tmpDir]);
                    return ['ok' => false, 'error' => 'Temp folder not writable ('.$tmpDir.')'];
                }
                $__probe = $tmpDir.'/.w';
                if (@file_put_contents($__probe, '1') === false) {
                    $zip->close();
                    \Log::error('ModuleCenter: temp write probe failed', ['tmpDir' => $tmpDir]);
                    return ['ok' => false, 'error' => 'Temp folder not writable ('.$tmpDir.')'];
                }
                @unlink($__probe);
                // Extract the entire ZIP to avoid path filtering edge cases
                $okBypass = $this->extractZipNormalized($zip, $tmpDir);
                $zip->close();
                if (!$okBypass) {
                    \Log::error('ModuleCenter: extractTo failed (single-folder bypass)', ['folder' => $folderName, 'tmpDir' => $tmpDir]);
                    if (!$debug) { $this->rrmdir($tmpDir); }
                    return ['ok' => false, 'error' => 'Extract failed'.($debug ? ' (tmp='.$tmpDir.')' : '')];
                }
                $modulesPath = base_path('modules');
                $destPath    = $modulesPath.'/'.$folderName;
                $srcRoot     = $tmpDir.'/'.$folderName;
                // Ensure source exists and has content before touching dest
                if (!is_dir($srcRoot) || !$this->dirHasContent($srcRoot)) {
                    $entries = @scandir($tmpDir) ?: [];
                    \Log::error('ModuleCenter: bypass source missing/empty', ['srcRoot' => $srcRoot, 'tmpDir' => $tmpDir, 'folder' => $folderName, 'entries' => $entries]);
                    if (!$debug) { $this->rrmdir($tmpDir); }
                    return ['ok' => false, 'error' => 'Extract produced empty folder'.($debug ? ' (tmp='.$tmpDir.')' : '')];
                }
                // Prefer replacing existing module folder by alias/name if present
                $alias = null; $installedName = null;
                $mfOnSrc = $this->findFileCaseInsensitive($srcRoot, ['module.json','manifest.json']);
                if ($mfOnSrc) {
                    try { $mjson = json_decode((string)file_get_contents($mfOnSrc), true); } catch (\Throwable $e) { $mjson = null; }
                    if (is_array($mjson)) { $alias = $mjson['alias'] ?? ($mjson['slug'] ?? null); $installedName = $mjson['name'] ?? null; }
                }
                $existing = $this->findInstalledFolderByAlias(base_path('modules'), $alias, $installedName);
                if ($existing) { $destPath = $modulesPath.'/'.$existing; }
                // Detect version from src
                $detectedVersion = $this->detectManifestVersion($mfOnSrc ?: null);
                // Stage copy first; do not delete existing dest until staging is ready
                $staging = $modulesPath.'/.staging_'.(($existing ?: $folderName)).'_'.uniqid();
                $this->rcopy($srcRoot, $staging);
                if (!is_dir($staging) || !$this->dirHasContent($staging)) {
                    \Log::error('ModuleCenter: staging copy failed', ['staging' => $staging]);
                    $this->rrmdir($staging);
                    if (!$debug) { $this->rrmdir($tmpDir); }
                    return ['ok' => false, 'error' => 'Staging failed'.($debug ? ' (tmp='.$tmpDir.')' : '')];
                }
                // Now backup and replace
                if (is_dir($destPath)) { $this->backupModuleDir($destPath, basename($destPath)); $this->rrmdir($destPath); }
                if (!@rename($staging, $destPath)) {
                    $this->rcopy($staging, $destPath);
                    $this->rrmdir($staging);
                }
                if (!$debug) { $this->rrmdir($tmpDir); }
                return ['ok' => true, 'name' => basename($destPath), 'installed_version' => $detectedVersion, 'tmp' => $debug ? $tmpDir : null];
            }

        $moduleJsonPath = null; $manifestJsonPath = null; $sample = [];
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $name = (string)$zip->getNameIndex($i);
            if ($i < 15) { $sample[] = $name; }
            $lc = strtolower($name);
            $base = strtolower(basename($name));
            if ($base === 'module.json' || substr($lc, -12) === 'module.json' || strpos($lc, '/module.json') !== false || strpos($lc, '\\module.json') !== false) {
                $moduleJsonPath = $name; break;
            }
            if ($base === 'manifest.json' || substr($lc, -13) === 'manifest.json' || strpos($lc, '/manifest.json') !== false || strpos($lc, '\\manifest.json') !== false) {
                $manifestJsonPath = $name; // keep looking for module.json first
            }
        }
        if (!$moduleJsonPath && $manifestJsonPath) { $moduleJsonPath = $manifestJsonPath; }
        if (!$moduleJsonPath) {
            \Log::error('ModuleCenter: module.json not found in ZIP. Sample entries: '.implode(', ', $sample));
            // Fallback: extract to temp and locate module.json on disk
            $tmpBase = storage_path('app/module_tmp');
            if (!is_dir($tmpBase)) { @mkdir($tmpBase, 0775, true); }
            $tmpDir = $tmpBase.'/zip_'.date('Ymd_His').'_'.uniqid();
            @mkdir($tmpDir, 0775, true);
            $ok2 = $zip->extractTo($tmpDir);
            if (!$ok2) { $ok2 = $this->extractZipNormalized($zip, $tmpDir); }
            $zip->close();
            if (!$ok2) {
                \Log::error('ModuleCenter: extractTo failed (no module.json path)', ['zip' => $zipPath, 'tmpDir' => $tmpDir]);
                if (!$debug) { $this->rrmdir($tmpDir); }
                return ['ok' => false, 'error' => 'Extract failed'.($debug ? ' (tmp='.$tmpDir.')' : '')];
            }
            $foundPath = $this->findFileCaseInsensitive($tmpDir, ['module.json','manifest.json']);
            if (!$foundPath) {
                $this->rrmdir($tmpDir);
                return ['ok' => false, 'error' => 'module.json not found in ZIP'];
            }
            $json = json_decode((string)file_get_contents($foundPath), true);
            $moduleName = $json['name'] ?? ($json['module'] ?? null);
            // Derive safe destination folder from the directory that contains module.json
            $rawFolder   = basename(dirname($foundPath));
            $folderName  = basename(str_replace(['\\'], '/', (string) $rawFolder));
            if ($folderName === '' || $folderName === '.' || $folderName === '..') {
                $attempt = $json['alias'] ?? $moduleName ?? '';
                $attempt = trim(basename(str_replace(['\\'], '/', (string)$attempt)));
                $attempt = preg_replace('/[^A-Za-z0-9._-]/', '', $attempt ?? '');
                if ($attempt !== '' && $attempt !== '.' && $attempt !== '..') {
                    $folderName = $attempt;
                } else {
                    Log::error('ModuleCenter: invalid folder (fallback-empty)', ['foundPath' => $foundPath, 'rawFolder' => $rawFolder, 'folderName' => $folderName]);
                    $this->rrmdir($tmpDir);
                    return ['ok' => false, 'error' => 'Invalid module folder in ZIP'];
                }
            }
            $modulesPath = base_path('modules');
            $destPath    = $modulesPath.'/'.$folderName;
            // Validate destination folder name (single directory name only)
            if (strpos($folderName, '/') !== false || strpos($folderName, '\\') !== false) {
                $this->rrmdir($tmpDir);
                return ['ok' => false, 'error' => 'Invalid module folder in ZIP'];
            }
            if (is_dir($destPath)) { $this->backupModuleDir($destPath, $folderName); $this->rrmdir($destPath); }
            $moduleRoot = dirname($foundPath);
            $this->rcopy($moduleRoot, $destPath);
            $this->rrmdir($tmpDir);
            return ['ok' => true, 'name' => $folderName];
        }
        $json = json_decode($zip->getFromName($moduleJsonPath), true);
        $moduleName = $json['name'] ?? ($json['module'] ?? null);
        if (!$moduleName) {
            $zip->close();
            return ['ok' => false, 'error' => 'Invalid module.json'];
        }

        // Always extract to a temporary folder, then copy the module root to modules/<name>
        $tmpBase = storage_path('app/module_tmp');
        if (!is_dir($tmpBase)) { @mkdir($tmpBase, 0775, true); }
        $tmpDir = $tmpBase.'/zip_'.date('Ymd_His').'_'.uniqid();
        @mkdir($tmpDir, 0775, true);
        $ok = $this->extractZipNormalized($zip, $tmpDir);
        $zip->close();
        if (!$ok) {
            $this->rrmdir($tmpDir);
            return ['ok' => false, 'error' => 'Extract failed'];
        }
        // If the ZIP extracted into exactly one top-level directory, use it directly
        $one = $this->singleTopLevelDir($tmpDir);
        if ($one) {
            $folderName  = trim(basename(str_replace(['\\'], '/', $one)));
            $modulesPath = base_path('modules');
            $destPath    = $modulesPath.'/'.$folderName;
            if ($folderName === '' || $folderName === '.' || $folderName === '..' || strpos($folderName, '/') !== false || strpos($folderName, '\\') !== false) {
                $this->rrmdir($tmpDir);
                return ['ok' => false, 'error' => 'Invalid module folder in ZIP'];
            }
            // Prefer replacing existing module folder by alias/name from the source manifest
            $alias = null; $installedName = null;
            $mfOnSrc = $this->findFileCaseInsensitive($one, ['module.json','manifest.json']);
            if ($mfOnSrc && file_exists($mfOnSrc)) {
                try { $mjson = json_decode((string)file_get_contents($mfOnSrc), true); } catch (\Throwable $e) { $mjson = null; }
                if (is_array($mjson)) { $alias = $mjson['alias'] ?? ($mjson['slug'] ?? null); $installedName = $mjson['name'] ?? null; }
            }
            $existing = $this->findInstalledFolderByAlias($modulesPath, $alias, $installedName);
            if ($existing) { $destPath = $modulesPath.'/'.$existing; }
            $detectedVersion = $this->detectManifestVersion($mfOnSrc ?: null);
            if (is_dir($destPath)) { $this->backupModuleDir($destPath, basename($destPath)); $this->rrmdir($destPath); }
            $this->rcopy($one, $destPath);
            $this->rrmdir($tmpDir);
            return ['ok' => true, 'name' => basename($destPath), 'installed_version' => $detectedVersion];
        }


        // Determine the on-disk path that contains module.json
        $relative = trim(str_replace(['\\'], '/', dirname($moduleJsonPath)), '/');
        $moduleRoot = $relative === '' || $relative === '.' ? $tmpDir : $tmpDir.'/'.$relative;
        // Determine destination folder name; prefer top-level folder, else alias or name
        $top        = $relative !== '' ? explode('/', $relative)[0] : '';
        $folderName = $top !== '' ? $top : ($json['alias'] ?? $moduleName);
        $folderName = trim(basename(str_replace(['\\'], '/', (string) $folderName)));

        $modulesPath = base_path('modules');
        $destPath    = $modulesPath.'/'.$folderName;

        // If an existing installed folder matches alias/name, prefer replacing that
        $existing = $this->findInstalledFolderByAlias($modulesPath, $json['alias'] ?? ($json['slug'] ?? null), $moduleName);
        if ($existing) { $destPath = $modulesPath.'/'.$existing; }

        // Validate destination folder name (single directory name only)
        if ($destPath === $modulesPath.'/' || strpos(basename($destPath), '/') !== false || strpos(basename($destPath), '\\') !== false) {
            $this->rrmdir($tmpDir);
            return ['ok' => false, 'error' => 'Invalid module folder in ZIP'];
        }

        // Detect version from moduleRoot
        $mf = $this->findFileCaseInsensitive($moduleRoot, ['module.json','manifest.json']);
        $detectedVersion = $this->detectManifestVersion($mf ?: null);

        // Backup and fully replace existing
        if (is_dir($destPath)) {
            $this->backupModuleDir($destPath, basename($destPath));
            $this->rrmdir($destPath);
        }
        $this->rcopy($moduleRoot, $destPath);
        $this->rrmdir($tmpDir);

        return ['ok' => true, 'name' => basename($destPath), 'installed_version' => $detectedVersion];
        } catch (\Throwable $e) {
            Log::error('Module installFromZip failed: '.$e->getMessage());
            return ['ok' => false, 'error' => 'Install error'];
        }
    }
    private function dirHasContent(string $dir): bool
    {
        if (!is_dir($dir)) { return false; }
        $list = @scandir($dir) ?: [];
        foreach ($list as $e) {
            if ($e === '.' || $e === '..') { continue; }
            return true;
        }
        return false;
    }


    private function backupModuleDir(string $dir, string $moduleName): void
    {
        $backupDir = storage_path('app/module_backups');
        if (!is_dir($backupDir)) { @mkdir($backupDir, 0775, true); }
        $zipPath = $backupDir.'/'.$moduleName.'_'.date('Ymd_His').'.zip';

        $zip = new ZipArchive();
        if ($zip->open($zipPath, ZipArchive::CREATE) !== true) {
            return; // best-effort backup
        }

        $dir = rtrim($dir, DIRECTORY_SEPARATOR);
        $baseLen = strlen(dirname($dir));
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($dir, \FilesystemIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );
        foreach ($iterator as $file) {
            $filePath = (string)$file;
            $localName = substr($filePath, $baseLen + 1);
            if (is_dir($filePath)) {
                $zip->addEmptyDir($localName);
            } else {
                $zip->addFile($filePath, $localName);
            }
        }
        $zip->close();
    }


    /**
     * If the temp extraction has exactly one top-level directory (ignoring __MACOSX), return its full path.
     */
    private function singleTopLevelDir(string $dir): ?string
    {
        $entries = @scandir($dir) ?: [];
        $dirs = [];
        foreach ($entries as $e) {
            if ($e === '.' || $e === '..' || $e === '__MACOSX') { continue; }
            $full = $dir.DIRECTORY_SEPARATOR.$e;
            if (is_dir($full)) { $dirs[] = $full; }
        }
        return count($dirs) === 1 ? $dirs[0] : null;
    }

    private function findFileCaseInsensitive(string $baseDir, array $names): ?string
    {
        $baseDir = rtrim($baseDir, DIRECTORY_SEPARATOR);
        if (!is_dir($baseDir)) { return null; }
        $priority = array_map('strtolower', $names);
        $entries = @scandir($baseDir) ?: [];
        foreach ($priority as $wanted) {
            foreach ($entries as $e) {
                if ($e === '.' || $e === '..') { continue; }
                if (strtolower($e) === $wanted) {
                    return $baseDir.DIRECTORY_SEPARATOR.$e;
                }
            }
        }
        // Last resort: recursive search preserving name priority
        foreach ($priority as $wanted) {
            $it = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($baseDir, \FilesystemIterator::SKIP_DOTS));
            foreach ($it as $file) {
                if (strtolower($file->getFilename()) === $wanted) {
                    return (string)$file->getPathname();
                }
            }
        }
        return null;
    }

    private function findInstalledFolderByAlias(string $modulesPath, ?string $alias, ?string $name): ?string
    {
        $targets = [];
        if (!empty($alias)) { $targets[] = $this->norm((string)$alias); }
        if (!empty($name))  { $targets[] = $this->norm((string)$name); }
        if (empty($targets)) { return null; }
        $dirs = @scandir($modulesPath) ?: [];
        foreach ($dirs as $d) {
            if ($d === '.' || $d === '..') { continue; }
            $mf = $modulesPath.'/'.$d.'/module.json';
            if (!file_exists($mf)) { continue; }
            try { $j = json_decode((string)file_get_contents($mf), true); } catch (\Throwable $e) { $j = null; }
            if (!is_array($j)) { continue; }
            $a = $this->norm((string)($j['alias'] ?? ($j['slug'] ?? ($j['name'] ?? $d))));
            if (in_array($a, $targets, true)) { return $d; }
        }
        return null;
    }

    private function parseVersionFromText(?string $raw): ?string
    {
        if (!is_string($raw) || $raw === '') { return null; }
        if (preg_match('/"\s*version\s*"\s*:\s*"([^\"]+)"/i', $raw, $m)) { return trim($m[1]); }
        if (preg_match('/version\s*[:=]\s*\"?([0-9][^\"\s,}]*)/i', $raw, $m)) { return trim($m[1]); }
        return null;
    }

    private function detectManifestVersion(?string $manifestPath): ?string
    {
        if (!$manifestPath || !file_exists($manifestPath)) { return null; }
        try {
            $raw = (string)file_get_contents($manifestPath);
            $json = json_decode($raw, true);
            if (is_array($json)) {
                $v = $json['version'] ?? ($json['Version'] ?? ($json['ver'] ?? ($json['module_version'] ?? null)));
                if ($v) { return (string)$v; }
                foreach ($json as $k => $vv) {
                    $clean = strtolower(preg_replace('/[^a-z0-9_]/', '', (string)$k));
                    if ($clean === 'version') { return is_scalar($vv) ? (string)$vv : null; }
                }
            }
            return $this->parseVersionFromText($raw);
        } catch (\Throwable $e) { return null; }
    }

    private function norm(string $s): string
    {
        $s = strtolower($s);
        return preg_replace('/[^a-z0-9]/', '', $s) ?? '';
    }

    private function extractZipNormalized(\ZipArchive $zip, string $tmpDir): bool
    {
        try {
            for ($i = 0; $i < $zip->numFiles; $i++) {
                $name = (string) $zip->getNameIndex($i);
                if ($name === '') { continue; }
                $p = str_replace(['\\'], '/', $name);
                $p = ltrim($p, '/');
                if ($p === '' || $p === '.' || $p === '..') { continue; }
                if (strpos($p, '__MACOSX/') === 0) { continue; }
                if (substr($p, -1) === '/') { @mkdir($tmpDir.'/'.$p, 0775, true); continue; }
                $target = $tmpDir.'/'.$p;
                $dir = dirname($target);
                if (!is_dir($dir)) { @mkdir($dir, 0775, true); }
                $stream = $zip->getStream($name);
                if ($stream !== false) {
                    $out = @fopen($target, 'w');
                    if ($out === false) { return false; }
                    @stream_copy_to_stream($stream, $out);
                    @fclose($stream); @fclose($out);
                } else {
                    $data = $zip->getFromIndex($i);
                    if ($data === false) { return false; }
                    @file_put_contents($target, $data);
                }
            }
            return true;
        } catch (\Throwable $e) {
            \Log::error('ModuleCenter: extractZipNormalized error', ['err' => $e->getMessage()]);
            return false;
        }
    }

    private function rcopy(string $src, string $dst): void
    {
        if (!is_dir($src)) { return; }
        if (!is_dir($dst)) { @mkdir($dst, 0775, true); }
        $it = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($src, \FilesystemIterator::SKIP_DOTS), \RecursiveIteratorIterator::SELF_FIRST);
        foreach ($it as $item) {
            $target = $dst.DIRECTORY_SEPARATOR.$it->getSubPathName();
            if ($item->isDir()) {
                if (!is_dir($target)) { @mkdir($target, 0775, true); }
            } else {
                @copy((string)$item->getPathname(), $target);
            }
        }
    }

    private function rrmdir(string $dir): void
    {
        if (!is_dir($dir)) { return; }
        $it = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($dir, \FilesystemIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::CHILD_FIRST
        );
        foreach ($it as $file) {
            $path = (string)$file->getPathname();
            if ($file->isDir()) { @rmdir($path); } else { @unlink($path); }
        }
        @rmdir($dir);
    }
}

